#pragma warning(disable:4786)

#include "MyGame.h"
#include "../engine/EntityRenderer.h"
#include "../engine/Mesh.h"
#include "../engine/Terrain.h"
#include "../engine/TerrainGen.h"
#include "../engine/camera/BasicCamera.h"
#include "../engine/resource/PictureBundle.h"

// ENGINE CONSTANTS
#include "EngineConstants.h"
#include "../utils/Random.h"

// game
#include "Deciders.h"
#include "../engine/Billboarder.h"
#include "../engine/TextRenderer.h"


#include <cassert>


// ------------------------------------------------------------------
//  basic stuff


CMyGame::CMyGame( CConfig& config )
:	CBaseGame( config ),
	mTerrainCamera( NULL ),
	mPool( NULL )
{
}

CMyGame::~CMyGame()
{
	delete mPool;
}

CCamera* CMyGame::createCamera()
{
	mTerrainCamera = new CBasicCamera();
	mTerrainCamera->getPosition() = D3DXVECTOR3( gcon::WORLD_X * 0.5f, gcon::WORLD_Y * 0.5f, 100.0f );

	return mTerrainCamera;
}

// ------------------------------------------------------------------
//  input


void CMyGame::onMouseLChange( bool pressed )
{
}

void CMyGame::onMouseRChange( bool pressed )
{
}

void CMyGame::onKeyChange( int key, bool pressed )
{
}

void CMyGame::onProcessInput()
{
	assert( mTerrainCamera );

	// update rotation from keys
	if( getKeyboard().isKeyPressed(VK_UP) )		mTerrainCamera->doPitch( 0.05f );
	if( getKeyboard().isKeyPressed(VK_DOWN) )	mTerrainCamera->doPitch( -0.05f );
	if( getKeyboard().isKeyPressed(VK_LEFT) )	mTerrainCamera->doYaw( 0.05f );
	if( getKeyboard().isKeyPressed(VK_RIGHT) )	mTerrainCamera->doYaw( -0.05f );
	if( getKeyboard().isKeyPressed('S') )		mTerrainCamera->doMove( 70.0f );
	if( getKeyboard().isKeyPressed('X') )		mTerrainCamera->doMove( -70.0f );
	if( getKeyboard().isKeyPressed('Z') )		mTerrainCamera->doStrafe( -40.0f );
	if( getKeyboard().isKeyPressed('C') )		mTerrainCamera->doStrafe( 40.0f );
	if( getKeyboard().isKeyPressed('Q') )		mTerrainCamera->doRise( 40.0f );
	if( getKeyboard().isKeyPressed('A') )		mTerrainCamera->doRise( -40.0f );
}


// ------------------------------------------------------------------
//  initialization


void CMyGame::onInitialize()
{
	// init stats
	mTotalBads = mTotalGoods = mTotalGods = 0;
	mKilledBads = mKilledGoods = mKilledGods = 0;
	// create deciders (they will be shared across many entities)
	mJumpDecider = new CJumpShotDecider();
	mDieDecider = new CDieDecider();
	
	mTerrainCamera->setTerrain( getTerrain() );

	int maxEntities = getEntities().getMaxEntities();
	mPool = new TPool( maxEntities );

	CPictureBundle& pb = CPictureBundle::getInstance();
	const CPicture& pic1 = *pb.getResourceById( CResourceId("Sprite.png") );
	
	CTerrainGen::fillTexture( getTerrain(), CResourceId("TerrainGray.png"), -50.0f, 100.0f );

	int count = maxEntities/2;

	int i;

	for( i = 0; i < count; ++i ) {
		CMyEntity& e = mPool->add();
		getEntities().add( e );
		e.getPosition() = D3DXVECTOR3( random::randf(gcon::WORLD_X), random::randf(gcon::WORLD_Y), 0.0f );
		e.mVelocity = D3DXVECTOR3( random::randfs(3.0f), random::randf(3.0f), 0.0f );
		e.getOldPosition() = e.getPosition();
		e.mDeciderCounter = rand()&15;
		e.mDecider = mJumpDecider; // normal decider
		e.setPicture( pic1 );

		// good or bad guy?
		int t = rand()&255;
		if( t < 255 ) {			// this is good guy
			e.setSize( 3.5f );	// he's small
			e.setColor( 0xFFC0C0C0 );	// he's gray
			e.mFlags = (CMyEntity::F_GRAVITY) | (CMyEntity::F_SMALL);
			++mTotalGoods;
		} else {				// this is bad guy - approx. every 1 of 256
			e.setSize( 8.0f );	// he's big
			e.setColor( 0xFFFFFF00 );	// he's yellow
			e.mFlags = CMyEntity::F_GRAVITY;
			++mTotalBads;
		}
		// god or not?
		if( (rand()%50) == 7 ) {	// approx. every 1 of 50
			mGods.push_back( &e );	// insert into god list
			e.mFlags |= CMyEntity::F_GOD;
			if( !mTotalGods )		// make the first god initially evil
				e.mFlags |= CMyEntity::F_EVILGOD;
			++mTotalGods;
		}
	}
}


// ------------------------------------------------------------------
//  game logic


void CMyGame::onUpdate()
{
	// is the game ended?
	if( mKilledBads >= mTotalBads || mKilledGoods >= mTotalGoods ) {
		// set NULL decider for all the entities, so that you can't
		// shoot them anymore, and the gods don't kill anymore.
		// everyone just wanders around.
		TPool::iterator it;
		for( it = mPool->begin(); it != mPool->end(); ++it ) {
			CMyEntity& e = *it;
			e.mDecider = NULL;
		}
	}
	
	// clear grid - remove rays from it and decrease "evil god damage"
	int n = mGrid.getSecsCount();
	for( int i = 0; i < n; ++i ) {
		mGrid.getSectorByIndex( i ).beginFrame();
	}
	
	// shoot (put ray into grid)
	if( getMouse().isMouseLPressed() ) {
		// construct a ray that goes from a camera to world
		CEntityRayPicker<TGrid>::putRay(
			mGrid,
			getCamera().getPosition(),
			getCamera().getWorldRay( getMouse().getMouseX(), getMouse().getMouseY() )
		);
	}
	
	// update entities
	TPool::iterator it;
	for( it = mPool->begin(); it != mPool->end();  ) {
		CMyEntity& e = *it;
		assert( e.mDeciderCounter >= 0 );
		e.update();
		if( e.mFlags & CMyEntity::F_KILLED ) {
			it = mPool->erase( it );
			getEntities().remove( e );
		} else {
			++it;
		}
	}
}


// ------------------------------------------------------------------
//  rendering

void CMyGame::onRenderBeforeAll()
{
	CBillboarder& bill = getBillboarder();
	bill.beginBillboards();
	
	// draw slightly transparent billboards above gods heads
	CResourceId god1( "God.png" ); // peaceful god picture
	CResourceId god2( "AngryGod.png" ); // evil god picture
	TEntityList::const_iterator it;
	for( it = mGods.begin(); it != mGods.end(); ++it ) {
		const CMyEntity& e = **it;
		bill.putEntityBill( (e.mFlags & (CMyEntity::F_EVILGOD)) ? god2 : god1, 0xC0FFFFFF, e, 20, 20 );
	}
	
	// indicators - one for your progress, other for evil gods progress
	float xx;
	xx = ((float)mKilledBads)/mTotalBads * 2.0f - 1.0f;
	bill.putScreenBill( CResourceId("Bar.png"), 0xFF00FF00, -1.0f, 0.8f, xx, 0.88f );
	xx = ((float)mKilledGoods)/mTotalGoods * 2.0f - 1.0f;
	bill.putScreenBill( CResourceId("Bar.png"), 0xFFFF0000, -1.0f, 0.9f, xx, 0.98f );
	
	bill.endBillboards();
}

void CMyGame::onRenderAfterWorld()
{
}

void CMyGame::onRenderAfterAll()
{
	// stats
	CTextRenderer& text = getTextRenderer();
	
	char buf[200];
	sprintf( buf, "Guys %i", mPool->size() );
	text.renderScreenText( 10, 25, buf, 0xFF808080 );
	sprintf( buf, "Bads\nGoods\nGods" );
	text.renderScreenText( 10, 40, buf, 0xFF808080 );
	sprintf( buf, "%i/%i\n%i/%i\n%i/%i",
		mKilledBads, mTotalBads,
		mKilledGoods, mTotalGoods,
		mKilledGods, mTotalGods
		);
	text.renderScreenText( 60, 40, buf, 0xFFC00000 );
	
	// win/lose text
	if( mKilledBads >= mTotalBads )
		text.renderScreenText( 240, 200, "You win!!!", 0xFFFFFF00 );
	if( mKilledGoods >= mTotalGoods )
		text.renderScreenText( 240, 200, "You lose...", 0xFFFF8000 );
}
