#include "Billboarder.h"

#include "Entity.h"
#include "Vertices.h"
#include "FrameTime.h"
#include "camera/Camera.h"
#include "resource/TextureBundle.h"
#include "resource/EffectBundle.h"
#include "../utils/DXHelpers.h"


CBillboarder::CBillboarder( CCamera const& camera, float const& alpha )
:	mIB( NULL ), mVB( NULL ),
	mCamera( camera ),
	mAlpha( alpha ),
	mSlots()
{
}

CBillboarder::~CBillboarder()
{
}



// ------------------------------------------------------------------
//  Billboards batching by textures into slots


CBillboarder::SSlot& CBillboarder::getSlot( const CResourceId& tex )
{
	IDirect3DTexture8* texture = CTextureBundle::getInstance().getResourceById( tex );
	TTextureSlotMap::iterator it = mSlots.find( texture );
	if( it == mSlots.end() ) {
		it = mSlots.insert( std::make_pair(texture,SSlot()) ).first;
	}
	return it->second;
}

void CBillboarder::beginBillboards()
{
	TTextureSlotMap::iterator it;
	for( it = mSlots.begin(); it != mSlots.end(); ++it ) {
		SSlot& s = it->second;
		s.mSize = 0;
	}
}

void CBillboarder::putWorldBill( const CResourceId& tex, D3DCOLOR color, const D3DXVECTOR3& pos, float width, float height )
{
	SSlot& slot = getSlot( tex );
	if( slot.mSize >= SSlot::CAPACITY )
		return;
	SBillboard& bill = slot.mBills[ slot.mSize++ ];
	bill.mColor = color;
	bill.mDrawMode = SBillboard::WORLD;
	bill.mWorld.mX = pos.x;
	bill.mWorld.mY = pos.y;
	bill.mWorld.mZ = pos.z;
	bill.mWorld.mWidth = width;
	bill.mWorld.mHeight = height;
}

void CBillboarder::putEntityBill( const CResourceId& tex, D3DCOLOR color, const CEntity& entity, float width, float height )
{
	SSlot& slot = getSlot( tex );
	if( slot.mSize >= SSlot::CAPACITY )
		return;
	SBillboard& bill = slot.mBills[ slot.mSize++ ];
	bill.mColor = color;
	bill.mDrawMode = SBillboard::ENTITY;
	bill.mEntity.mEntity = &entity;
	bill.mEntity.mWidth = width;
	bill.mEntity.mHeight = height;
}

void CBillboarder::putScreenBill( const CResourceId& tex, D3DCOLOR color, float x1, float y1, float x2, float y2 )
{
	SSlot& slot = getSlot( tex );
	if( slot.mSize >= SSlot::CAPACITY )
		return;
	SBillboard& bill = slot.mBills[ slot.mSize++ ];
	bill.mColor = color;
	bill.mDrawMode = SBillboard::SCREEN;
	bill.mScreen.mX1 = x1;
	bill.mScreen.mY1 = y1;
	bill.mScreen.mX2 = x2;
	bill.mScreen.mY2 = y2;
}

void CBillboarder::endBillboards()
{
}


// ------------------------------------------------------------------
//  DX part

// Buffers capacity in billboards.
static const int BUFFERS_CAPACITY = 2000;


HRESULT CBillboarder::onCreateDevice( IDirect3DDevice8* device )
{
	CDXObject::onCreateDevice( device );
	
	// IB
	assert( !mIB );
	getDevice().CreateIndexBuffer(
		BUFFERS_CAPACITY * 6 * sizeof(short),
		0,
		D3DFMT_INDEX16,
		D3DPOOL_MANAGED,
		&mIB
	);
	assert( mIB );

	short* ib = NULL;
	mIB->Lock( 0, 0, (BYTE**)&ib, 0 );
	assert( ib );
	for( int i = 0; i < BUFFERS_CAPACITY; ++i ) {
		short base = i * 4;
		ib[0] = base;
		ib[1] = base+1;
		ib[2] = base+2;
		ib[3] = base;
		ib[4] = base+2;
		ib[5] = base+3;
		ib += 6;
	}
	mIB->Unlock();
	
	return S_OK;
}

HRESULT CBillboarder::onLostDevice()
{
	CDXObject::onLostDevice();
	dx::assertRelease( mVB );
	return S_OK;
}

HRESULT CBillboarder::onResetDevice()
{
	CDXObject::onResetDevice();
	
	// VB
	assert( !mVB );
	getDevice().CreateVertexBuffer(
		BUFFERS_CAPACITY * 4 * sizeof(SVertexXyzDiffuseTex1),
		D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY,
		FVF_XYZ_DIFFUSE_TEX1,
		D3DPOOL_DEFAULT,
		&mVB
	);
	assert( mVB );
	
	return S_OK;
}

HRESULT CBillboarder::onDestroyDevice()
{
	CDXObject::onDestroyDevice();
	
	dx::assertRelease( mIB );
	
	return S_OK;
}


// ------------------------------------------------------------------
//  Rendering


void CBillboarder::render( CFrameTime const& frameTime )
{
	D3DXMATRIX viewMatrix = mCamera.getCameraMatrix();
	viewMatrix._41 = viewMatrix._42 = viewMatrix._43 = 0.0f;
	const D3DXVECTOR3& camPos = mCamera.getPosition();
	float camNearMul = mCamera.getZNear() * 1.001f;

	ID3DXEffect& effect = *CEffectBundle::getInstance().getResourceById( CResourceId("Billboard.sha") );
	
	int bufferPosition = 0;
	SVertexXyzDiffuseTex1* vb = NULL;
	
	TTextureSlotMap::iterator it;
	for( it = mSlots.begin(); it != mSlots.end(); ++it ) {
		
		// begin slot
		IDirect3DTexture8* texture = it->first;
		SSlot& slot = it->second;
		if( slot.mSize < 1 )
			continue;
		effect.SetTexture( "texBase", texture );
		UINT passes, p;
		effect.Begin( &passes, 0 );
		
		// lock buffer
		int slotVertices = slot.mSize*4;
		if( bufferPosition == 0 || bufferPosition+slotVertices >= BUFFERS_CAPACITY*4 ) {
			mVB->Lock( 0, 0, (BYTE**)&vb, D3DLOCK_DISCARD );
			bufferPosition = 0;
		} else {
			mVB->Lock( bufferPosition*sizeof(SVertexXyzDiffuseTex1), slotVertices*sizeof(SVertexXyzDiffuseTex1), (BYTE**)&vb, D3DLOCK_NOOVERWRITE );
		}

		// fill with slot billboards
		for( int i = 0; i < slot.mSize; ++i ) {
			SBillboard& bill = slot.mBills[i];

			// fill
			D3DXVECTOR3 pos;
			float width2, height2;
			const CEntity* entity;
			
			switch( bill.mDrawMode ) {
			case SBillboard::WORLD:
				pos.x = bill.mWorld.mX;
				pos.y = bill.mWorld.mY;
				pos.z = bill.mWorld.mZ;
				width2 = bill.mWorld.mWidth * 0.5f;
				height2 = bill.mWorld.mHeight * 0.5f;
				renderBill( vb, pos, width2, height2, bill.mColor, viewMatrix );
				vb += 4;
				break;
			case SBillboard::ENTITY:
				entity = bill.mEntity.mEntity;
				pos = entity->getOldPosition() + (entity->getPosition() - entity->getOldPosition()) * mAlpha;
				width2 = bill.mEntity.mWidth * 0.5f;
				height2 = bill.mEntity.mHeight * 0.5f;
				pos.z += entity->getSize() * 0.6f + height2;
				renderBill( vb, pos, width2, height2, bill.mColor, viewMatrix );
				vb += 4;
				break;
			case SBillboard::SCREEN:
				{
					pos = mCamera.getPosition();
					D3DXVECTOR3 cUL = pos + mCamera.getWorldRay( bill.mScreen.mX1, bill.mScreen.mY1 ) * camNearMul;
					D3DXVECTOR3 cUR = pos + mCamera.getWorldRay( bill.mScreen.mX2, bill.mScreen.mY1 ) * camNearMul;
					D3DXVECTOR3 cDL = pos + mCamera.getWorldRay( bill.mScreen.mX1, bill.mScreen.mY2 ) * camNearMul;
					D3DXVECTOR3 cDR = pos + mCamera.getWorldRay( bill.mScreen.mX2, bill.mScreen.mY2 ) * camNearMul;
					vb->p = cUL; vb->diffuse = bill.mColor; vb->tu = 0.0f; vb->tv = 0.0f; ++vb;
					vb->p = cUR; vb->diffuse = bill.mColor; vb->tu = 1.0f; vb->tv = 0.0f; ++vb;
					vb->p = cDR; vb->diffuse = bill.mColor; vb->tu = 1.0f; vb->tv = 1.0f; ++vb;
					vb->p = cDL; vb->diffuse = bill.mColor; vb->tu = 0.0f; vb->tv = 1.0f; ++vb;
				}
				break;
			}
		}
		
		// unlock
		mVB->Unlock();

		// draw
		getDevice().SetIndices( mIB, bufferPosition );
		getDevice().SetStreamSource( 0, mVB, sizeof(SVertexXyzDiffuseTex1) );
		for( p = 0; p < passes; ++p ) {
			effect.Pass( p );
			getDevice().DrawIndexedPrimitive( D3DPT_TRIANGLELIST, 0, slotVertices, 0, slotVertices/2 );
		}
		bufferPosition += slotVertices;
		effect.End();
	}
}


void CBillboarder::renderBill( SVertexXyzDiffuseTex1* vb, const D3DXVECTOR3& pos, float width2, float height2, DWORD color, const D3DXMATRIX& viewMat )
{
	D3DXVECTOR3 cornerUL( -width2, height2,  0.0f );
	D3DXVECTOR3 cornerUR(  width2, height2,  0.0f );
	D3DXVECTOR3 cornerDL( -width2, -height2, 0.0f );
	D3DXVECTOR3 cornerDR(  width2, -height2, 0.0f );
	D3DXVECTOR3 cUL, cUR, cDL, cDR;
	D3DXVec3TransformCoord( &cUL, &cornerUL, &viewMat );
	D3DXVec3TransformCoord( &cDL, &cornerDL, &viewMat );
	D3DXVec3TransformCoord( &cUR, &cornerUR, &viewMat );
	D3DXVec3TransformCoord( &cDR, &cornerDR, &viewMat );
	
	// 4 particle corners
	vb->p = pos + cUL;
	vb->diffuse = color;
	vb->tu = 0.0f; vb->tv = 0.0f;
	++vb;
	vb->p = pos + cUR;
	vb->diffuse = color;
	vb->tu = 1.0f; vb->tv = 0.0f;
	++vb;
	vb->p = pos + cDR;
	vb->diffuse = color;
	vb->tu = 1.0f; vb->tv = 1.0f;
	++vb;
	vb->p = pos + cDL;
	vb->diffuse = color;
	vb->tu = 0.0f; vb->tv = 1.0f;
	++vb;
}

