#pragma warning( disable:4786 )
#include "PictureBundle.h"

#include "../../utils/Errors.h"
#include "../../utils/DXHelpers.h"

#include <cassert>
#include <d3dx8tex.h>


CPictureBundle::CPictureBundle( IDirect3DDevice8& device, std::string predir, int texSizeX, int texSizeY, int baseSize )
:	mDevice( device ),
	mPreDir( predir ),
	mTexSizeX( texSizeX ),
	mTexSizeY( texSizeY ),
	mBaseSize( baseSize ),
	mTexture( NULL ),
	mSurface( NULL ),
	mAreasX( texSizeX / baseSize ),
	mAreasY( texSizeY / baseSize ),
	mFreeAreas( NULL ),
	mDefaultPicture( 0.0f, 0.0f, 1.0f, 1.0f )
{
	assert( mAreasX > 1 && mAreasY > 1 );
	mFreeAreas = new bool[ mAreasX * mAreasY ];
	clear();

	HRESULT hres = getDevice().CreateTexture(
		mTexSizeX, mTexSizeY, 0, 0, D3DFMT_A8R8G8B8, D3DPOOL_MANAGED, &mTexture
	);
	if( !SUCCEEDED( hres ) ) {
		THROW_DXERROR( hres, "failed to create picture manager texture" );
	}
	assert( mTexture );
	
	hres = mTexture->GetSurfaceLevel( 0, &mSurface );
	if( !SUCCEEDED( hres ) ) {
		THROW_DXERROR( hres, "failed to get surface of picture manager texture" );
	}
	assert( mSurface );
}

CPictureBundle::~CPictureBundle()
{
	dx::release( mSurface );
	dx::release( mTexture );

	// HACK: in order to clear resources
	clear();
	CONSOLE.write( "picture bundle destroyed" );

	assert( mFreeAreas );
	delete[] mFreeAreas;
}


void CPictureBundle::clear()
{
	CStorageResourceBundle<CPicture>::clear();

	int n = mAreasX * mAreasY;
	for( int i = 0; i < n; ++i )
		mFreeAreas[i] = true;
	mFreeAreasCount = n;
}


CPicture* CPictureBundle::loadResourceById( CResourceId const& id )
{
	HRESULT hres;
	
	std::string fileName = CPictureBundle::getInstance().getPreDir() + id.getUniqueName();
	
	// get file info
	
	D3DXIMAGE_INFO info;
	hres = D3DXGetImageInfoFromFile( fileName.c_str(), &info );
	if( FAILED(hres) ) {
		CON_WARNING.write( "can't read file " + fileName );
		return new CPicture( 0, 0, 1, 1 );
	}
	if( info.ResourceType != D3DRTYPE_TEXTURE ) {
		CON_WARNING.write( "file " + fileName + " does not contain a texture" );
		return new CPicture( 0, 0, 1, 1 );
	}
	
	// calc needed size
	
	int sizeX = mBaseSize;
	int sizeY = mBaseSize;
	int areasX = 1;
	int areasY = 1;
	
	if( sizeX + sizeX/2 < info.Width ) {
		sizeX *= 2;
		areasX = 2;
	}
	if( sizeY + sizeY/2 < info.Height ) {
		sizeY *= 2;
		areasY = 2;
	}
	
	// find empty space
	int ax, ay;
	bool ok = findSpace( areasX, areasY, ax, ay );
	if( !ok ) {
		CON_WARNING.write( "no texture space left for " + fileName );
		return new CPicture( 0, 0, 1, 1 );
	}
	
	int ix1 = ax * mBaseSize;
	int iy1 = ay * mBaseSize;
	float x1 = (float)ax / mAreasX;
	float y1 = (float)ay / mAreasY;
	float x2 = (float)(ax+areasX) / mAreasX;
	float y2 = (float)(ay+areasY) / mAreasY;
	
	// load the texture and copy it into big one
	IDirect3DTexture8* tex = NULL;
	hres = D3DXCreateTextureFromFileEx( &getDevice(), fileName.c_str(), sizeX, sizeY, 1,
		0, D3DFMT_A8R8G8B8, D3DPOOL_SYSTEMMEM, D3DX_DEFAULT, D3DX_DEFAULT, 0,
		NULL, NULL, &tex );
	if( FAILED(hres) ) {
		CON_WARNING.write( "can't load texture from " + fileName );
		return new CPicture( 0, 0, 1, 1 );
	}
	assert( tex );
	
	IDirect3DSurface8* surf = NULL;
	hres = tex->GetSurfaceLevel( 0, &surf );
	if( FAILED(hres) ) {
		CON_WARNING.write( "can't get surface level from texture" );
		return new CPicture( 0, 0, 1, 1 );
	}
	
	RECT rect;
	rect.left = 1;
	rect.right = sizeX-1;
	rect.top = 1;
	rect.bottom = sizeY-1;
	POINT point;
	point.x = ix1+1;
	point.y = iy1+1;
	getDevice().CopyRects( surf, &rect, 1, mSurface, &point );
	
	dx::release( surf );
	dx::release( tex );
	
	D3DXFilterTexture( mTexture, NULL, 0, D3DX_DEFAULT );
	
	// return
	
	CONSOLE.write( "picture loaded '" + id.getUniqueName() + "'" );

	return new CPicture( x1, y1, x2, y2 );
}

void CPictureBundle::touchResource( CPicture& resource )
{
}

void CPictureBundle::clearResource( CPicture& resource )
{
	assert( &resource );
	delete &resource;
}

bool CPictureBundle::findSpace( int areasX, int areasY, int& ax, int& ay )
{
	if( mFreeAreasCount < areasX * areasY )
		return false;

	for( ay = 0; ay < mAreasY-areasY+1; ++ay ) {
		for( ax = 0; ax < mAreasX-areasX+1; ++ax ) {
			int x, y;
			for( y = 0; y < areasY; ++y ) {
				for( x = 0; x < areasX; ++x ) {
					if( !mFreeAreas[ (ay+y) * mAreasX + ax+x ] )
						goto _notFree;
				}
			}
			// free - mark it
			for( y = 0; y < areasY; ++y ) {
				for( x = 0; x < areasX; ++x ) {
					mFreeAreas[ (ay+y) * mAreasX + ax+x ] = false;
					--mFreeAreasCount;
				}
			}
			return true;
_notFree:
			// not free
			;
		}
	}
}

