/**
 *  @file utils/Config.h
 *  Configuration files support.
 */
#pragma warning(disable:4786)
#ifndef __CONFIG_H__
#define __CONFIG_H__

#include <string>
#include <vector>
#include <list>

/**
 *	Configuration files support.
 *
 *	Item in config file is represented by a name, a value and
 *  (optionally) a section it belongs. Names and sections are case
 *  sensitive. Section name is written in a separate line between '['
 *  and ']' brackets; all items till next section belong to this
 *  section. All items before the very first section belong to all
 *  sections. Comments are written after '#' or ';' till end of line.
 *
 *  Program can optionally modify config file values (or append new
 *  ones), and they can be written back to the file (if writing
 *  is not disabled in the constructor).
 *
 *  Global game config file can be accesed via CEngine.
 *  @see CEngine.
 */
class CConfig {
public:
	/// Vector of int's.
	typedef std::vector<int>	TIntVector;
	/// Vector of float's.
	typedef std::vector<float>	TFloatVector;
	/// Vector of std::string's.
	typedef std::vector<std::string> TStringVector;
public:
	/**
	 *  Constructor.
	 *  @param readName Name of the config file.
	 *  @param write    If true, then modified/appended values are
	 *                  written back to the file.
	 */
	CConfig( const std::string& readName = "default.cfg", bool write = false );
	/**
	 *  Constructor.
	 *  @param readName     Name of the config file.
	 *  @param writeName    Name of new config file that contains the
	 *                      original values and the modified/added ones.
	 */
	CConfig( const std::string& readName, const std::string& writeName );
	/**
	 *  Destructor.
	 *  Writing of modified/added values happens upon destruction only!
	 */
	~CConfig();

	
	/**
	 *  Read string from config.
	 *  @param section	The section to take item from.
	 *  @param name		The name of the item.
	 *  @param def		The default value (if needed item isn't present).
	 */
	std::string  readS( const std::string& section, const std::string& name, const std::string& def );
	/**
	 *  Read integer from config.
	 *  @param section	The section to take item from.
	 *  @param name		The name of the item.
	 *  @param def		The default value (if needed item isn't present).
	 */
	int 		 readI( const std::string& section, const std::string& name, int def );
	/**
	 *  Read float from config.
	 *  @param section	The section to take item from.
	 *  @param name		The name of the item.
	 *  @param def		The default value (if needed item isn't present).
	 */
	float		 readF( const std::string& section, const std::string& name, float def );

	/**
	 *  Read strings from config (eg. name = abc ghty llkfj).
	 *  @param section	The section to take item from.
	 *  @param name		The name of the item.
	 */
	TStringVector	readSn( const std::string& section, const std::string& name );
	/**
	 *  Read ints from config (eg. name = 1 -7 124).
	 *  @param section	The section to take item from.
	 *  @param name		The name of the item.
	 */
	TIntVector		readIn( const std::string& section, const std::string& name );
	/**
	 *  Read floats from config (eg. name = 14.5 -3.0 3.1415927).
	 *  @param section	The section to take item from.
	 *  @param name		The name of the item.
	 */
	TFloatVector	readFn( const std::string& section, const std::string& name );

	
	/**
	 *  Sets item value as a string. Modifies existing one or appends.
	 *  @param section	The section of the item.
	 *  @param name		The name of the item.
	 *  @param val		The new value.
	 */
	void modifyS( const std::string& section, const std::string& name, const std::string& val );
	/**
	 *  Sets item value as an integer. Modifies existing one or appends.
	 *  @param section	The section of the item.
	 *  @param name		The name of the item.
	 *  @param val		The new value.
	 */
	void modifyI( const std::string& section, const std::string& name, int val );
	/**
	 *  Sets item value as a float. Modifies existing one or appends.
	 *  @param section	The section of the item.
	 *  @param name		The name of the item.
	 *  @param val		The new value.
	 */
	void modifyF( const std::string& section, const std::string& name, float val );

private:
	struct SLine { // one config line
		std::string 	mName; // name
		std::string 	mData; // value
		std::string 	mOther; // comments
		bool			mWrite; // false for includes
	};
	typedef std::list<SLine*>	TLineList;

private:
	// methods
	static void prepareLine( const std::string& in, std::string& name, std::string& data, std::string& other );
	static std::string prepareSection( const std::string& in );
	SLine* findString( const std::string& section, const std::string& name, TLineList::iterator* iter );
	SLine* insertLine( const TLineList::iterator& iter, const std::string& name, const std::string& data );
	void addFile( const std::string& readName, bool write );
	// data
	TLineList	mLines;
	std::string mWriteName; // will be written to
	bool	mDirty; // is modified?
	bool	mWrite; // is writing allowed?
};

#endif

