/**
 *  @file game/Decider.h
 *  IDecider interface.
 */
#ifndef __DECIDER_H
#define __DECIDER_H

#include "Home.h"
#include "MyEntity.h"
#include "../utils/Random.h"

/**
 *  Entity decider (AI) interface.
 *
 *  Entity logic happens through it's decider - each entity has a
 *  decider that gets called at some times (based on counter). A
 *  decider should do entity logic (eg. alter it's velocity, size,
 *  color, sprite, kill it) and set a new counter value.
 *
 *  Deciders can be shared accross many entities.
 *
 *  @see CGameEntity.
 */
class IDecider {
public:
	virtual ~IDecider() { }

	/**
	 *  Perform logic for an entity.
	 *  @param e The entity.
	 */
	virtual void decideFor( CMyEntity& e ) const = 0;
};

/*class CFollowCursor
{
public:
	virtual void route( COverlordEntity& e ) {
		D3DXVECTOR3 d = e.mTarget - e.getPosition();

		float distance = D3DXVec3LengthSq( &d ) * 0.05f;
		distance = min( distance, 5.0f );

		D3DXVec3Normalize( &d, &d );
		d *= distance;

		e.mVelocity = d;
	}
};*/

class CFollowOverlord : public IDecider
{
public:
	CFollowOverlord( COverlordEntity const& overlord, float maxSpeed, float epsilon = 0.0005f )
		: mOverlord( overlord ), mMaxSpeed( maxSpeed ), mEpsilon( epsilon ) {}

	virtual void decideFor( CMyEntity& e ) const {
		D3DXVECTOR3 d = mOverlord.getPosition() - e.getPosition() +
			D3DXVECTOR3( random::randfs(10.0f), random::randf(10.0f), 0.0f );

		float distance = D3DXVec3LengthSq( &d ) * mEpsilon;
		distance = min( distance, mMaxSpeed );

		D3DXVec3Normalize( &d, &d );
		d *= distance;

		e.mVelocity = d;
	}

private:
	COverlordEntity const&	mOverlord;
	float					mMaxSpeed;
	float					mEpsilon;
};


class CGoHome : public IDecider
{
public:
	CGoHome( CHome& home, IDecider& next, float maxSpeed, D3DCOLOR color ) 
		: mHome( home ), mNext( next ), mMaxSpeed( maxSpeed ), mColor( color ) {}

	virtual void decideFor( CMyEntity& e ) const {
		D3DXVECTOR3 d = mHome.mPosition - e.getPosition();

		float distance = D3DXVec3LengthSq( &d );
		if( distance < 10000 )
		{
//			e.mVelocity = D3DXVECTOR3( 0.0f, 0.0f, 0.0f );
			e.setColor( mColor );
			e.mDecider = &mNext;
			e.mDeciderCounter = 1;

			mHome.grow( e.mFood );
			e.mStrength = e.mFood;
			e.mFood = 0.0f;
			return;
		}

		D3DXVec3Normalize( &d, &d );
		d *= mMaxSpeed;

		e.mVelocity = d;
	}
public:
	D3DCOLOR	mColor;

private:
	CHome&		mHome;
	IDecider&	mNext;
	float		mMaxSpeed;

};

#endif