#pragma warning( disable:4786 )
#include "EntityRenderer.h"

#include "Entity.h"
#include "Vertices.h"
#include "FrameTime.h"
#include "camera/Camera.h"
#include "resource/EffectBundle.h"
#include "resource/PictureBundle.h"
#include "../utils/DXHelpers.h"


CEntityRenderer::CEntityRenderer( int maxEntities, CCamera const& camera, float const& alpha )
:	mMaxEntities( maxEntities ),
	mEntityCount( 0 ),
	mIB( NULL ),
	mVB( NULL ),
	mCamera( camera ),
	mAlpha( alpha )
{
	mSprites = new SSpriteData[ maxEntities ];
	mEntityPtrs = new CEntity*[ maxEntities ];
}

CEntityRenderer::~CEntityRenderer()
{
	delete[] mSprites;
	delete[] mEntityPtrs;
}


void CEntityRenderer::add( CEntity& entity )
{
	SSpriteInfo info( mSprites[mEntityCount], mEntityCount );
	entity.getSpriteInfo() = info;
	mEntityPtrs[ mEntityCount ] = &entity;
	++mEntityCount;
}

void CEntityRenderer::remove( CEntity& entity )
{
	int index = entity.getSpriteInfo().index;
	--mEntityCount;
	mSprites[index] = mSprites[mEntityCount];
	mEntityPtrs[index] = mEntityPtrs[mEntityCount];
	mEntityPtrs[index]->getSpriteInfo().data = &mSprites[index];
	mEntityPtrs[index]->getSpriteInfo().index = index;
}


HRESULT CEntityRenderer::onCreateDevice( IDirect3DDevice8* device )
{
	CDXObject::onCreateDevice( device );
	return S_OK;
}

HRESULT CEntityRenderer::onLostDevice()
{
	CDXObject::onLostDevice();
	dx::release( mIB );
	dx::release( mVB );
	return S_OK;
}

HRESULT CEntityRenderer::onResetDevice()
{
	CDXObject::onResetDevice();
	assert( !mIB );
	getDevice().CreateIndexBuffer(
		BUFFER_SIZE * 6 * sizeof(short), D3DUSAGE_WRITEONLY,
		D3DFMT_INDEX16, D3DPOOL_DEFAULT,
		&mIB
	);
	assert( mIB );
	short* ib = NULL;
	mIB->Lock( 0, 0, (BYTE**)&ib, 0 );
	assert( ib );
	for( int i = 0; i < BUFFER_SIZE; ++i ) {
		short base = i * 4;
		ib[0] = base;
		ib[1] = base+1;
		ib[2] = base+2;
		ib[3] = base;
		ib[4] = base+2;
		ib[5] = base+3;
		ib += 6;
	}
	mIB->Unlock();
	
	assert( !mVB );
	getDevice().CreateVertexBuffer(
		BUFFER_SIZE * 4 * sizeof( TVertex ), D3DUSAGE_WRITEONLY | D3DUSAGE_DYNAMIC,
		FVF_XYZ_DIFFUSE_TEX1, D3DPOOL_DEFAULT, &mVB
	);
	assert( mVB );
	return S_OK;
}

HRESULT CEntityRenderer::onDestroyDevice()
{
	CDXObject::onDestroyDevice();
	return S_OK;
}




void CEntityRenderer::render( CFrameTime const& frameTime )
{
	if( mEntityCount <= 0 )
		return;

	// ----------------------------------------------------------
	//  entities

	static D3DXVECTOR3 cornerUL( -1.0f, 1.0f,  0.0f );
	static D3DXVECTOR3 cornerUR(  1.0f, 1.0f,  0.0f );
	static D3DXVECTOR3 cornerDL( -1.0f, -1.0f, 0.0f );
	static D3DXVECTOR3 cornerDR(  1.0f, -1.0f, 0.0f );

	ID3DXEffect& effect = *CEffectBundle::getInstance().getResourceById( CResourceId("Sprite.sha") );
	IDirect3DTexture8& texture = CPictureBundle::getInstance().getUnderlyingTexture();
	effect.SetTexture( "texBase", &texture );
	
	UINT passes;
	effect.Begin( &passes, 0 );

	D3DXMATRIX viewMatrix = mCamera.getCameraMatrix();
	viewMatrix._41 = viewMatrix._42 = viewMatrix._43 = 0.0f;

	D3DXVECTOR3 cUL, cUR, cDL, cDR;
	D3DXVec3TransformCoord( &cUL, &cornerUL, &viewMatrix );
	D3DXVec3TransformCoord( &cDL, &cornerDL, &viewMatrix );
	D3DXVec3TransformCoord( &cUR, &cornerUR, &viewMatrix );
	D3DXVec3TransformCoord( &cDR, &cornerDR, &viewMatrix );

	// draw
	getDevice().SetStreamSource( 0, mVB, sizeof(TVertex) );

	TVertex* vbb = NULL;

	int sCount = mEntityCount;
	const SSpriteData* sIt = mSprites;
	float alpha = mAlpha;

	int renderCount = (sCount + BUFFER_SIZE-1) / BUFFER_SIZE;
	for( int renders = 0; renders < renderCount; ++renders ) {

		int renderSize = BUFFER_SIZE;
		if( renders == renderCount-1 )
			renderSize = sCount;

		int bufferStart = 0;

		int portionCount = (renderSize + LOCK_SIZE-1) / LOCK_SIZE;
		for( int portion = 0; portion < portionCount; ++portion ) {

			int portionSize = LOCK_SIZE;
			if( portion == portionCount-1 )
				portionSize = renderSize - (portion) * LOCK_SIZE;

			if( !portion )
				mVB->Lock( 0, portionSize*4*sizeof(TVertex), (BYTE**)&vbb, D3DLOCK_DISCARD );
			else
				mVB->Lock( bufferStart*4*sizeof(TVertex), portionSize*4*sizeof(TVertex), (BYTE**)&vbb, D3DLOCK_NOOVERWRITE );

			int i = portionSize-1;
			do {
				SSpriteData const& p = *sIt;
				
				// calc
				D3DXVECTOR3 pos = p.oldPosition + (p.position - p.oldPosition) * alpha;
				
				D3DCOLOR color = p.color;
				float size = p.size;
				const CPicture& pic = *p.picture;
				
				// 4 particle corners
				vbb->p = pos + cUL * size;
				vbb->diffuse = color;
				vbb->tu = pic.getX1(); vbb->tv = pic.getY1();
				++vbb;
				vbb->p = pos + cUR * size;
				vbb->diffuse = color;
				vbb->tu = pic.getX2(); vbb->tv = pic.getY1();
				++vbb;
				vbb->p = pos + cDR * size;
				vbb->diffuse = color;
				vbb->tu = pic.getX2(); vbb->tv = pic.getY2();
				++vbb;
				vbb->p = pos + cDL * size;
				vbb->diffuse = color;
				vbb->tu = pic.getX1(); vbb->tv = pic.getY2();
				++vbb;
				
				++sIt;
				--i;
			} while( i >= 0 );
			
			// unlock
			mVB->Unlock();

			// render
			for( int p = 0; p < passes; ++p ) {
				effect.Pass( p );
				getDevice().SetIndices( mIB, bufferStart * 4 );
				getDevice().DrawIndexedPrimitive( D3DPT_TRIANGLELIST, 0, portionSize * 4, 0, portionSize * 2 );
			}

			bufferStart += portionSize;
			sCount -= portionSize;

		} // for portion

	} // for render


	effect.End();
}
