/**
 *  @file engine/Selector.h
 *  Utility to select rectangular world areas.
 */
#ifndef __SELECTOR_H
#define __SELECTOR_H

#include <d3dx8math.h>
#include <cassert>

#include "Grid.h"
#include "../game/EngineConstants.h"


struct SSelectionSector {
public:
	enum eSelection { NONE = 0, FULL = 1, PART = 2 };
public:
	SSelectionSector() : mSelection(NONE) { }

	void beginFrame() { mSelection = NONE; }

	bool isSelected( float x, float y ) const {
		if( mSelection == NONE )
			return false;
		if( mSelection == FULL )
			return true;
		return x >= mSelX1 && x <= mSelX2 && y >= mSelY1 && y <= mSelY2;
	}

public:
	eSelection	mSelection;
	float		mSelX1, mSelY1;
	float		mSelX2, mSelY2;
};


static const float RAYPICK_SMALL_DIR = 0.1f;


/**
 *  Utility to select rectangular world areas.
 *  Grid sector class must have the following public members:
 *		- eSelection mSelection; enum eSelection { NONE = 0, FULL = 1, PART = 2 };
 *		- float mSelX1, mSelY1, mSelX2, mSelY2;
 *
 *  @param GRID Grid class.
 *
 *  @see CGrid.
 */
template<typename GRID>
class CSelector {
public:
	/**
	 *  Puts selection into grid.
	 *  For all sectors that are touched by a selection, puts the selection into
	 *  the sector. Does not modify untouched sectors.
	 *  @param grid The grid.
	 *  @param x1 Minimum X coordinate.
	 *  @param y1 Minimum Y coordinate.
	 *  @param x2 Maximum X coordinate.
	 *  @param y2 Maximum Y coordinate.
	 */
	static void putSelection( GRID& grid, float x1, float y1, float x2, float y2 )
	{
		if( x1 > x2 ) {
			float t = x1;
			x1 = x2;
			x2 = t;
		}
		if( y1 > y2 ) {
			float t = y1;
			y1 = y2;
			y2 = t;
		}
		int ix1, iy1, ix2, iy2;
		grid.position2Index( x1, y1, ix1, iy1 );
		grid.position2Index( x2, y2, ix2, iy2 );
		int ix, iy;
		for( ix = ix1; ix <= ix2; ++ix ) {
			GRID::TSector& sector = grid.getSectorByIndex( ix, iy1 );
			sector.mSelection = GRID::TSector::PART;
			sector.mSelX1 = x1; sector.mSelY1 = y1;
			sector.mSelX2 = x2; sector.mSelY2 = y2;
		}
		for( iy = iy1+1; iy < iy2; ++iy ) {
			GRID::TSector& sector1 = grid.getSectorByIndex( ix1, iy );
			sector1.mSelection = GRID::TSector::PART;
			sector1.mSelX1 = x1; sector1.mSelY1 = y1;
			sector1.mSelX2 = x2; sector1.mSelY2 = y2;
			for( ix = ix1; ix <= ix2; ++ix ) {
				GRID::TSector& sector = grid.getSectorByIndex( ix, iy );
				sector.mSelection = GRID::TSector::FULL;
			}
			GRID::TSector& sector2 = grid.getSectorByIndex( ix2, iy );
			sector2.mSelection = GRID::TSector::PART;
			sector2.mSelX1 = x1; sector2.mSelY1 = y1;
			sector2.mSelX2 = x2; sector2.mSelY2 = y2;
		}
		for( ix = ix1; ix <= ix2; ++ix ) {
			GRID::TSector& sector = grid.getSectorByIndex( ix, iy2 );
			sector.mSelection = GRID::TSector::PART;
			sector.mSelX1 = x1; sector.mSelY1 = y1;
			sector.mSelX2 = x2; sector.mSelY2 = y2;
		}
	}
};


#endif
