/**
 * @file engine/camera/Camera.h
 * Base camera.
 */

#ifndef __CAMERA_H
#define __CAMERA_H

#include <d3dx8math.h>
#include <d3d8.h>

class CFrameTime;

// ------------------------------------------------------------------
//  Camera

/**
 * Basic camera.
 * Has only the matrices: camera (as object), view (inverse of camera), projection.
 * Provides methods for setting the matrices for DX, and some convenience methods.
 */
class CCamera {
public:
	CCamera();
	virtual ~CCamera();

	virtual void update( CFrameTime const& frameTime ) { }

	void	setProjectionParams( float fov, float aspect, float znear, float zfar );
	void	setDeviceTransforms( IDirect3DDevice8& device, bool view = true, bool proj = false );

	const D3DXMATRIX& getProjectionMatrix() const { return mProjectionMatrix; }
	const D3DXMATRIX& getViewMatrix() const { return mViewMatrix; }
	const D3DXMATRIX& getCameraMatrix() const { return mCameraMatrix; }

	/// Gets camera position in world space.
	D3DXVECTOR3 getPosition() const { return D3DXVECTOR3(mCameraMatrix._41,mCameraMatrix._42,mCameraMatrix._43); }
	/// Gets FOV
	float getFOV() const { return mFOV; }
	/// Gets near Z plane
	float getZNear() const { return mZNear; }
	/// Gets far Z plane
	float getZFar() const { return mZFar; }

	/**
	 *  Gets world space ray from camera's viewing plane coordinates.
	 *  @param x X coordinate (-1 at left, 1 at right).
	 *  @param y Y coordinate (-1 at top, 1 at bottom).
	 *  @return Ray in world space (NOT unit length).
	 */
	D3DXVECTOR3 getWorldRay( float x, float y ) const;
	/**
	 *  Gets camera space ray from camera's viewing plane coordinates.
	 *  @param x X coordinate (-1 at left, 1 at right).
	 *  @param y Y coordinate (-1 at top, 1 at bottom).
	 *  @return Ray in camera space (NOT unit length).
	*/
	D3DXVECTOR3 getCameraRay( float x, float y ) const;
	
protected:
	void		updateMatrices(); // from camera matrix
	D3DXMATRIX	mCameraMatrix;
	
private:
	D3DXMATRIX	mProjectionMatrix;
	D3DXMATRIX	mViewMatrix;

	float		mFOV;
	float		mZNear;
	float		mZFar;
	float		mViewHalfWidth; // at distance 1
	float		mViewHalfHeight; // at distance 1
};



#endif
