/**
 *  @file game/MyGame.h
 *  Your game application code.
 */

#ifndef __MY_GAME_H
#define __MY_GAME_H

#include "MyEntity.h"
#include "GodlessEntity.h"
#include "CloudEntity.h"
#include "BallEntity.h"
#include "../engine/BaseGame.h"
#include "../engine/Grid.h"
#include "../utils/FixedPool.h"
#include "../utils/Singleton.h"

typedef CFixedPool<CBallEntity*> TBallEntityPtrPool;

class CBallSector {
public:
	static const int MAX_BALLS;

	CBallSector();
	CBallSector( const CBallSector& s );
	~CBallSector();

	void assignToBall( CGodlessEntity& entity );

	void addBall( CBallEntity& ball );
	void removeBall( CBallEntity& ball );

private:
	TBallEntityPtrPool* mBalls;
};
typedef CGrid<CBallSector,16,16> TBallGrid;


class CBasicCamera;
/**
 *  Your game application class.
 */
class CMyGame : public CBaseGame, public CSingleton<CMyGame>
{
	static int GODLESS_COUNT;
	static int BALL_COUNT;
	static const int CLOUD_COUNT;
public:
	static const float STONE_RADIUS;
	static const float STONE_RADIUS_2;

	typedef CFixedPool<CMyEntity>		TPool;
	typedef CFixedPool<CGodlessEntity>	TGodlessEntityPool;
	typedef CFixedPool<CBallEntity>		TBallEntityPool;
	typedef CFixedPool<CCloudEntity>		TCloudEntityPool;

public:
	static void init( CConfig& config ) {
		CMyGame* game = new CMyGame( config ); assert( game );
		assignInstance( *game );
	}
	virtual ~CMyGame();

	TBallGrid*		getGrid() { return mGrid; }


protected:
	/// Receive left mouse button change.
	virtual void	onMouseLChange( bool pressed );
	/// Receive right mouse button change.
	virtual void	onMouseRChange( bool pressed );
	/// Receive key state change.
	virtual void	onKeyChange( int key, bool pressed );
	
	/**
	 *  Initialize the game.
	 */
	virtual	void onInitialize();
	
	/**
	 *  Process input.
	 *  The engine calls this method for processing input.
	 */
	virtual void onProcessInput();

	/**
	 *  Execute game logic.
	 *  The engine calls this method when game update time comes.
	 */
	virtual void onUpdate();

	/**
	 *  Perform additional rendering.
	 *  The engine calls this method before rendering anything.
	 */
	virtual void onRenderBeforeAll();

	/**
	 *  Perform additional rendering.
	 *  The engine calls this method after rendering all world
	 *  objects.
	 */
	virtual void onRenderAfterWorld();

	/**
	 *  Perform additional rendering.
	 *  The engine calls this method after rendering everything.
	 */
	virtual void onRenderAfterAll();

protected:
	virtual CCamera* createCamera();

private:
	CMyGame( CConfig& config );

	static CMyGame* createInstance() {
		assert( !"Must be initialized first" );
		return 0;
	}

	static void deleteInstance( CMyGame& instance ) {
		delete &instance;
	}

	void putBall();
	std::string getString( int value );
	void orientCamera();
	void initCamera();

	friend CSingleton<CMyGame>;	

private:
	CBasicCamera*	mTerrainCamera;

public:
	TPool*			mPool;
	TGodlessEntityPool* mGodlessEntityPool;
	TBallEntityPool*	mBallEntityPool;
	TCloudEntityPool*	mCloudEntityPool;
	int				mCurrentBall;
	TBallGrid*		mGrid;
	CTerrain*		mTerrainWater;

	int mGodlessDeadCount;
	int mGodFound;
	bool mM2Pressed;

	bool mGameStarted;
};

#endif