/**
 *  @file engine/Grid.h
 *  Grid template.
 */
#ifndef __GRID_H
#define __GRID_H

#include <cassert>
#include "../game/EngineConstants.h"
#include "../utils/MathUtil.h"

/**
 *  Grid - a 2D world partition.
 *
 *  A grid divides world space into fixed grid of rectangular sectors.
 *  World is divided on XY plane, so each sector represents a whole
 *  vertical "column" of the world.
 *
 *  Needed sector interface:
 *		- void beginFrame();
 *		- void endFrame();
 *
 *  @param SECTOR Sector class.
 *  @param SECS_X Number of sectors along world X axis. Should be a power of 2.
 *  @param SECS_Y Number of sectors along world Y axis. Should be a power of 2.
 */
template <typename SECTOR,int SECS_X=32,int SECS_Y=32>
class CGrid {
public:
	/// Sector type (from template argument).
	typedef SECTOR TSector;
public:
	/**
	 *  Constructor.
	 *  @param t Sector object used to initialize all the sectors.
	 */
	CGrid( SECTOR const& t = SECTOR() )
		:	mStepX( (gcon::WORLD_X)/SECS_X ), mStepY( (gcon::WORLD_Y)/SECS_Y ),
			mInvStepX( (float)SECS_X/(gcon::WORLD_X) ), mInvStepY( (float)SECS_Y/(gcon::WORLD_Y) )
	{
		assert( SECS_X > 0 && SECS_Y > 0 );

		int n = SECS_X * SECS_Y;
		mSectors = new SECTOR*[ n ];
		for( int q = 0; q < n; ++q )
			mSectors[q] = new SECTOR( t );
	}

	/// Destructor.
	~CGrid() {
		int n = SECS_X * SECS_Y;
		for( int q = 0; q < n; ++q )
			delete mSectors[q];
		delete[] mSectors;
	}

	/// Returns number of sectors along X axis.
	int getSecsX() const { return SECS_X; }
	/// Returns number of sectors along Y axis.
	int getSecsY() const { return SECS_Y; }
	/// Returns total number of sectors.
	int getSecsCount() const { return SECS_X * SECS_Y; }

	/// Returns sector size along X axis.
	float getStepX() const { return mStepX; }
	/// Returns sector size along Y axis.
	float getStepY() const { return mStepY; }
	/// Returns inverse sector size along X axis.
	float getInvStepX() const { return mInvStepX; }
	/// Returns inverse sector size along Y axis.
	float getInvStepY() const { return mInvStepY; }

	/// Returns read-only sector by world space position XY.
	const SECTOR& getSectorByPosition( float x, float y ) const {
		return getSectorByIndex( position2Index(x,y) );
	}
	/// Returns sector by world space position XY.
	SECTOR& getSectorByPosition( float x, float y ) {
		return getSectorByIndex( position2Index(x,y) );
	}
	/// Returns read-only sector by it's XY index.
	const SECTOR& getSectorByIndex( int ix, int iy ) const {
		assert( ix >= 0 && ix < SECS_X );
		assert( iy >= 0 && iy < SECS_Y );
		return *mSectors[ iy * SECS_X + ix ];
	}
	/// Returns sector by it's XY index.
	SECTOR& getSectorByIndex( int ix, int iy ) {
		assert( ix >= 0 && ix < SECS_X );
		assert( iy >= 0 && iy < SECS_Y );
		return *mSectors[ iy * SECS_X + ix ];
	}
	/// Returns read-only sector by it's absolute index.
	const SECTOR& getSectorByIndex( int i ) const {
		assert( i >= 0 && i < SECS_X * SECS_Y );
		return *mSectors[ i ];
	}
	/// Returns sector by it's absolute index.
	SECTOR& getSectorByIndex( int i ) {
		assert( i >= 0 && i < SECS_X * SECS_Y );
		return *mSectors[ i ];
	}

	/// Returns sector's absolute index from world XY position.
	int position2Index( float x, float y ) const {
		int ix = math::round( x * mInvStepX - 0.5f );
		int iy = math::round( y * mInvStepY - 0.5f );
		assert( ix >= 0 && ix < SECS_X );
		assert( iy >= 0 && iy < SECS_Y );
		return iy*SECS_X+ix;
	}
	/// Returns sector's XY index from world XY position.
	void position2Index( float x, float y, int& ix, int& iy ) const {
		ix = math::round( x * mInvStepX - 0.5f );
		iy = math::round( y * mInvStepY - 0.5f );
		assert( ix >= 0 && ix < SECS_X );
		assert( iy >= 0 && iy < SECS_Y );
	}
	
private:
	const float	mStepX;
	const float	mStepY;
	const float	mInvStepX;
	const float	mInvStepY;

	SECTOR**	mSectors;
};

#endif
