#ifndef __BASE_GAME_H
#define __BASE_GAME_H

#include <cassert>
#include "DXObject.h"
#include "input/Keyboard.h"
#include "input/Mouse.h"
#include "FrameTime.h"

class CConfig;
class CEntityRenderer;
class CTerrain;
class CCamera;
class CBillboarder;
class CTextRenderer;
class CAppContext;


/**
 *  Base game class.
 */
class CBaseGame : public CDXObject
{
public:
	explicit CBaseGame( CConfig& config );
	virtual ~CBaseGame();

	// Returns read-only config.
	const CConfig& getConfig() const { return mConfig; }
	/// Returns config.
	CConfig& getConfig() { return mConfig; }

	// Returns read-only keyboard.
	const CKeyboard& getKeyboard() const { assert( mKeyboard ); return *mKeyboard; }

	// Returns read-only mouse.
	const CMouse& getMouse() const { assert( mMouse ); return *mMouse; }

	/// Returns read-only entity renderer.
	const CEntityRenderer& getEntities() const { assert( mEntities ); return *mEntities; }
	/// Returns entity renderer.
	CEntityRenderer& getEntities() { assert( mEntities ); return *mEntities; }

	/// Returns read-only terrain.
	const CTerrain& getTerrain() const { assert( mTerrain ); return *mTerrain; }
	/// Returns terrain.
	CTerrain& getTerrain() { assert( mTerrain ); return *mTerrain; }
	
	/// Returns read-only camera.
	const CCamera& getCamera() const { assert( mCamera ); return *mCamera; }
	/// Returns camera.
	CCamera& getCamera() { assert( mCamera ); return *mCamera; }

	/// Returns read-only billboarder.
	const CBillboarder& getBillboarder() const { assert( mBillboarder ); return *mBillboarder; }
	/// Returns billboarder.
	CBillboarder& getBillboarder() { assert( mBillboarder ); return *mBillboarder; }

	/// Returns read-only text renderer.
	const CTextRenderer& getTextRenderer() const { assert( mTextRenderer ); return *mTextRenderer; }
	/// Returns text renderer.
	CTextRenderer& getTextRenderer() { assert( mTextRenderer ); return *mTextRenderer; }

	/// Returns read-only visible objects container.
	const CDXObjectContainer& getObjects() const { return mObjects; }
	/// Returns visible objects container.
	CDXObjectContainer& getObjects() { return mObjects; }

	CFrameTime const& getHackFrameTimeForAnimation() { return mHackFrameTimeForAnimation; }


	// CDXObjectContainer
	virtual HRESULT onCreateDevice( IDirect3DDevice8* device );
	virtual HRESULT onLostDevice();
	virtual HRESULT onResetDevice();
	virtual HRESULT onDestroyDevice();
	virtual void render( CFrameTime const& frameTime );


	// interface methods for CGameApp class instance
	void update( CFrameTime const& frameTime );
	void setAppContext( CAppContext const& appContext );
	void setKeyboard( CKeyboard& keyboard );
	void setMouse( CMouse& mouse );

protected:
	virtual CTerrain* createTerrain();
	virtual CCamera* createCamera();

	virtual CEntityRenderer* createEntityRenderer();
	virtual CBillboarder* createBillboarder();
	virtual CTextRenderer* createTextRenderer();

protected:
	/// Receive left mouse button change.
	virtual void	onMouseLChange( bool pressed );
	/// Receive right mouse button change.
	virtual void	onMouseRChange( bool pressed );
	/// Receive key state change.
	virtual void	onKeyChange( int key, bool pressed );
	
	/**
	 *  Initialize the game.
	 */
	virtual void	onInitialize();
	
	/**
	 *  Process input.
	 *  The engine calls this method for processing input.
	 */
	virtual void	onProcessInput( const CFrameTime& frameTime );

	/**
	 *  Execute game logic.
	 *  The engine calls this method when game update time comes.
	 */
	virtual void	onUpdate();

	/**
	 *  Perform additional rendering.
	 *  The engine calls this method before rendering anything.
	 */
	virtual void	onRenderBeforeAll();

	/**
	 *  Perform additional rendering.
	 *  The engine calls this method after rendering all world
	 *  objects.
	 */
	virtual void	onRenderAfterWorld();

	/**
	 *  Perform additional rendering.
	 *  The engine calls this method after rendering everything.
	 */
	virtual void	onRenderAfterAll();

private:
	class CKeyboardListener : public IKeyboardListener
	{
	public:
		CKeyboardListener( CBaseGame& delegate ) : mDelegate( delegate ) {}

		virtual void onKeyChange( int key, bool pressed ) {
			mDelegate.onKeyChange( key, pressed );
		}

	private:
		CBaseGame&	mDelegate;
	};
	friend CKeyboardListener;

	class CMouseListener : public IMouseListener
	{
	public:
		CMouseListener( CBaseGame& delegate ) : mDelegate( delegate ) {}

		virtual void onMouseLChange( bool pressed ) {
			mDelegate.onMouseLChange( pressed );
		}
		virtual void onMouseRChange( bool pressed ) {
			mDelegate.onMouseRChange( pressed );
		}

	private:
		CBaseGame&	mDelegate;
	};
	friend CMouseListener;

private:
	void initialize();

private:
	CConfig&			mConfig;

	CAppContext const*	mAppContext;
	CKeyboard const*	mKeyboard;
	CMouse const*		mMouse;

	CKeyboardListener	mKeyboardListener;
	CMouseListener		mMouseListener;

	CCamera*			mCamera;
	
	CEntityRenderer*	mEntities;
	CTerrain*			mTerrain;
	CBillboarder*		mBillboarder;
	CTextRenderer*		mTextRenderer;

	CDXObjectContainer	mObjects;

	float				mTimeSinceLastActualUpdate;
	float				mAlpha;

	CFrameTime			mHackFrameTimeForAnimation;

/*	TPool*			mPool;
	CBasicCamera*	mCamera;*/
};

#endif
