/**
 *  @file engine/Billboarder.h
 *  Billboards rendering.
 */

#pragma warning(disable:4786)
#ifndef __BILLBOARDER_H
#define __BILLBOARDER_H

#include <d3d8.h>
#include <d3dx8math.h>
#include <map>
#include <vector>
#include "DXObject.h"
#include "resource/ResourceId.h"


class CEntity;
class CCamera;
class CFrameTime;
struct SVertexXyzDiffuseTex1;


/**
 *  Billboard renderer.
 *
 *  Renders billboards (aka sprites, or screen-aligned textured
 *  quads). The only methods you should be interested in are
 *  beginBillboards(), endBillboards() and all the put???Bill().
 *  The rest is handled by the engine.
 *
 *  All the put???Bill() methods should be called between
 *  beginBillboards() and endBillboards().
 */
class CBillboarder : public CDXObject {
public:
	CBillboarder( CCamera const& camera, float const& alpha );
	virtual ~CBillboarder();
	
	virtual HRESULT onCreateDevice( IDirect3DDevice8* device );
	virtual HRESULT onLostDevice();
	virtual HRESULT onResetDevice();
	virtual HRESULT onDestroyDevice();

	/// Begins billboards.
	void	beginBillboards();
	/**
	 *  Puts world space billboard.
	 *  @param tex		Texture ID.
	 *  @param color	Color.
	 *  @param pos		World space position.
	 *  @param width	Width (in world units).
	 *  @param height	Height (in world units).
	 */
	void	putWorldBill( const CResourceId& tex, D3DCOLOR color, const D3DXVECTOR3& pos, float width, float height );
	/**
	 *  Puts billboard above given entity.
	 *  @param tex		Texture ID.
	 *  @param color	Color.
	 *  @param entity	Entity rederence.
	 *  @param width	Width (in world units).
	 *  @param height	Height (in world units).
	 */
	void	putEntityBill( const CResourceId& tex, D3DCOLOR color, const CEntity& entity, float width, float height );
	/**
	 *  Puts screen space billboard. The billboard occupies given
	 *  portion of the screen, no matter what texture size is.
	 *  @param tex		Texture ID.
	 *  @param color	Color.
	 *  @param x1		Left x coordinate (-1 at left, 1 at right).
	 *  @param y1		Top y coordinate (-1 at top, 1 at bottom).
	 *  @param x2		Right x coordinate (-1 at left, 1 at right).
	 *  @param y2		Bottom y coordinate (-1 at top, 1 at bottom).
	 */
	void	putScreenBill( const CResourceId& tex, D3DCOLOR color, float x1, float y1, float x2, float y2 );
	/// Ends billboards.
	void	endBillboards();

	virtual void render( CFrameTime const& frameTime );

private:
	struct SWorldBill {
		float	mX, mY, mZ; // VC does not allow D3DXVECTOR3 here
		float	mWidth, mHeight;
	};
	struct SEntityBill {
		const CEntity*	mEntity;
		float	mWidth, mHeight;
	};
	struct SScreenBill {
		float	mX1, mY1;
		float	mX2, mY2;
	};
	struct SBillboard {
	public:
		enum eDrawMode { WORLD = 0, ENTITY = 1, SCREEN = 2 };
	public:
		D3DCOLOR	mColor;
		eDrawMode	mDrawMode;
		union {
			SWorldBill			mWorld;
			SEntityBill			mEntity;
			SScreenBill			mScreen;
		};
	};

	struct SSlot {
	public:
		enum { CAPACITY = 128 };
		SSlot() : mSize(0) { }
	public:
		int			mSize;
		SBillboard	mBills[CAPACITY];
	};

	typedef std::map<IDirect3DTexture8*,SSlot> TTextureSlotMap;

private:
	SSlot&	getSlot( const CResourceId& tex );
	static void renderBill( SVertexXyzDiffuseTex1* vb, const D3DXVECTOR3& pos, float width2, float height2, DWORD color, const D3DXMATRIX& viewMat );

private:
	IDirect3DIndexBuffer8*	mIB;
	IDirect3DVertexBuffer8*	mVB;

	CCamera const&			mCamera;
	float const&			mAlpha;

	TTextureSlotMap			mSlots;
};

#endif
