#pragma warning( disable:4786 )

#include "Terrain.h"
#include "Vertices.h"
#include "FrameTime.h"
#include "../utils/DXHelpers.h"
#include "../utils/MathUtil.h"
#include "../game/EngineConstants.h"
#include "resource/EffectBundle.h"
#include "resource/TextureBundle.h"
#include <cassert>


static const float CELL_SIZE_X = (float)(gcon::WORLD_X) / (CTerrain::CELLS_X-1);
static const float CELL_SIZE_Y = (float)(gcon::WORLD_Y) / (CTerrain::CELLS_Y-1);
static const float CELL_INV_SIZE_X = 1.0f / CELL_SIZE_X;
static const float CELL_INV_SIZE_Y = 1.0f / CELL_SIZE_Y;

// ------------------------------------------------------------------
//  constr/destr

CTerrain::CTerrain( const CResourceId& effect, const CResourceId& texture, float tuTiles, float tvTiles )
:	mEffect( effect ), mTexture( texture ), mTuTiles(tuTiles), mTvTiles(tvTiles),
	mIB( NULL ),
	mIBRect( NULL ),
	mVB( NULL ),
	mVBRect( NULL ),
	mMinAltitude( 0.0f ), mMaxAltitude( 100.0f )
{
	mTuScale = mTuTiles / CELLS_X;
	mTvScale = mTvTiles / CELLS_Y;
	
	mVertexCount = CELLS;
	mTriCount = mVertexCount * 2;

	mAltitudes = new float[ CELLS ];

	clearQuad();
}


CTerrain::~CTerrain()
{
	assert( mAltitudes );
	delete[] mAltitudes;
}


// ------------------------------------------------------------------
//  DX part

HRESULT CTerrain::onCreateDevice( IDirect3DDevice8* device )
{
	CDXObject::onCreateDevice( device );

	// IB
	assert( !mIB );
	getDevice().CreateIndexBuffer(
		mTriCount * 3 * sizeof(short),
		0,
		D3DFMT_INDEX16,
		D3DPOOL_MANAGED,
		&mIB
	);
	assert( mIB );
	fillIB();

	return S_OK;
}

HRESULT CTerrain::onLostDevice()
{
	CDXObject::onLostDevice();
	dx::assertRelease( mVB );
	dx::assertRelease( mVBRect );
	dx::assertRelease( mIBRect );
	return S_OK;
}

HRESULT CTerrain::onResetDevice()
{
	CDXObject::onResetDevice();

	// VB
	assert( !mVB );
	getDevice().CreateVertexBuffer(
		mVertexCount * sizeof(SVertexXyzTex1),
		D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY,
		FVF_XYZ_TEX1,
		D3DPOOL_DEFAULT,
		&mVB
	);
	assert( mVB );
	fillVB();

	assert( !mVBRect );
	getDevice().CreateVertexBuffer(
		mVertexCount * sizeof(SVertexXyz),
		D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY,
		FVF_XYZ,
		D3DPOOL_DEFAULT,
		&mVBRect
	);
	assert( mVBRect );
	assert( !mIBRect );
	getDevice().CreateIndexBuffer(
		mTriCount * 3 * sizeof(short),
		D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY,
		D3DFMT_INDEX16,
		D3DPOOL_DEFAULT,
		&mIBRect
	);
	assert( mIBRect );
	
	return S_OK;
}

HRESULT CTerrain::onDestroyDevice()
{
	CDXObject::onDestroyDevice();

	dx::assertRelease( mIB );

	return S_OK;
}

void CTerrain::fillIB()
{
	short* idx;
	mIB->Lock( 0, 0, (BYTE**)&idx, 0 );
	for( int y = 0; y < CELLS_Y-1; ++y ) {
		for( int x = 0; x < CELLS_X-1; ++x ) {
			int base = y * CELLS_X + x;
			*idx++ = base;
			*idx++ = base+1;
			*idx++ = base+CELLS_X+1;
			*idx++ = base;
			*idx++ = base+CELLS_X+1;
			*idx++ = base+CELLS_X;
		}
	}
	mIB->Unlock();
}

void CTerrain::fillVB()
{
	mMinAltitude = 1.0e6f;
	mMaxAltitude = -1.0e6f;

	SVertexXyzTex1 *curr;
	mVB->Lock( 0, 0, (BYTE**)&curr, D3DLOCK_DISCARD );
	
	const float* alt = mAltitudes;
	
	float yCrd = 0.0f;
	float tvCrd = 0.0f;
	for( int y = 0; y < CELLS_Y; ++y ) {
		float xCrd = 0.0f;
		float tuCrd = 0.0f;
		for( int x = 0; x < CELLS_X; ++x ) {
			float z = *alt;
			if( z < mMinAltitude )
				mMinAltitude = z;
			if( z > mMaxAltitude )
				mMaxAltitude = z;

			// pos
			curr->p.x = xCrd;
			curr->p.y = yCrd;
			curr->p.z = z;
			// uv
			curr->tu = tuCrd;
			curr->tv = tvCrd;
			
			++curr;
			++alt;

			xCrd += CELL_SIZE_X;
			tuCrd += mTuScale;
		}

		yCrd += CELL_SIZE_Y;
		tvCrd += mTvScale;
	}

	mVB->Unlock();
}


// ------------------------------------------------------------------
//  render

void CTerrain::render( CFrameTime const& frameTime )
{
	getDevice().SetIndices( mIB, 0 );
	getDevice().SetStreamSource( 0, mVB, sizeof(SVertexXyzTex1) );

	ID3DXEffect& effect = *CEffectBundle::getInstance().getResourceById( mEffect );
	IDirect3DTexture8& texture = *CTextureBundle::getInstance().getResourceById( mTexture );
	effect.SetTexture( "texBase", &texture );
	
	UINT p, passes;
	effect.Begin( &passes, 0 );
	for( p = 0; p < passes; ++p ) {
		effect.Pass( p );
		getDevice().DrawIndexedPrimitive( D3DPT_TRIANGLELIST, 0, mVertexCount, 0, mTriCount );
	}
	effect.End();
}

void CTerrain::clearQuad()
{
	mRectVertexCount = 0;
}

void CTerrain::prepareQuad( float x1, float y1, float x2, float y2, float addZ )
{
	clearQuad();
	if( x1 == x2 || y1 == y2 )
		return;

	float t;
	if( x1 > x2 ) {
		t = x2; x2 = x1; x1 = t;
	}
	if( y1 > y2 ) {
		t = y2; y2 = y1; y1 = t;
	}

	assert( x1 >= 0.0f && x2 < (gcon::WORLD_X) );
	assert( y1 >= 0.0f && y2 < (gcon::WORLD_Y) );

	//
	// fill VB

	int vertsX=0, vertsY=0;
	float x, y;
	float dx;
	float dx1 = (CELL_SIZE_X-fmodf(x1,CELL_SIZE_X));

	SVertexXyz *curr;
	mVBRect->Lock( 0, 0, (BYTE**)&curr, D3DLOCK_DISCARD );

	//
	// top row vertices

	x = x1; dx = dx1;
	do {
		curr->p.x = x;
		curr->p.y = y1;
		curr->p.z = getAltitude( x, y1 ) + addZ;
		++curr;
		++mRectVertexCount;
		++vertsX;
		x += dx;
		dx = CELL_SIZE_X;
	} while( x < x2 );
	curr->p.x = x2;
	curr->p.y = y1;
	curr->p.z = getAltitude( x2, y1 ) + addZ;
	++curr;
	++mRectVertexCount;
	++vertsX;

	//
	// rows of inner vertices at y gridlines

	++vertsY;

	y = y1 + CELL_SIZE_Y - fmodf(y1,CELL_SIZE_Y);
	while( y < y2 ) {
		x = x1; dx = dx1;
		do {
			curr->p.x = x;
			curr->p.y = y;
			curr->p.z = getAltitude( x, y ) + addZ;
			++curr;
			++mRectVertexCount;
			x += dx;
			dx = CELL_SIZE_X;
		} while( x < x2 );
		curr->p.x = x2;
		curr->p.y = y;
		curr->p.z = getAltitude( x2, y ) + addZ;
		++curr;
		++mRectVertexCount;

		y += CELL_SIZE_Y;
		++vertsY;
	}

	//
	// last row vertices

	x = x1; dx = dx1;
	do {
		curr->p.x = x;
		curr->p.y = y2;
		curr->p.z = getAltitude( x, y2 ) + addZ;
		++curr;
		++mRectVertexCount;
		x += dx;
		dx = CELL_SIZE_X;
	} while( x < x2 );
	curr->p.x = x2;
	curr->p.y = y2;
	curr->p.z = getAltitude( x2, y2 ) + addZ;
	++curr;
	++mRectVertexCount;

	++vertsY;

	mVBRect->Unlock();

	assert( mRectVertexCount <= mVertexCount );
	assert( mRectVertexCount == vertsX * vertsY );

	//
	// fill IB

	mRectIndexCount = 0;
	short* idx;
	mIBRect->Lock( 0, 0, (BYTE**)&idx, D3DLOCK_DISCARD );
	for( int iy = 0; iy < vertsY-1; ++iy ) {
		for( int ix = 0; ix < vertsX-1; ++ix ) {
			int base = iy * vertsX + ix;
			*idx++ = base;
			*idx++ = base+1;
			*idx++ = base+vertsX+1;
			*idx++ = base;
			*idx++ = base+vertsX+1;
			*idx++ = base+vertsX;
			mRectIndexCount += 6;
		}
	}
	mIBRect->Unlock();
}


void CTerrain::renderQuad( D3DCOLOR color )
{
	getDevice().SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );
	getDevice().SetRenderState( D3DRS_SRCBLEND, D3DBLEND_SRCALPHA );
	getDevice().SetRenderState( D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA );
	getDevice().SetRenderState( D3DRS_ZWRITEENABLE, FALSE );
	getDevice().SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
	
	getDevice().SetRenderState( D3DRS_TEXTUREFACTOR, color );
	getDevice().SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_SELECTARG1 );
	getDevice().SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TFACTOR );
	getDevice().SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1 );
	getDevice().SetTextureStageState( 0, D3DTSS_ALPHAARG1, D3DTA_TFACTOR );

	if( mRectVertexCount > 0 ) {
		getDevice().SetVertexShader( FVF_XYZ );
		getDevice().SetIndices( mIBRect, 0 );
		getDevice().SetStreamSource( 0, mVBRect, sizeof(SVertexXyz) );
		getDevice().DrawIndexedPrimitive( D3DPT_TRIANGLELIST, 0, mRectVertexCount, 0, mRectIndexCount/3 );
	}

	getDevice().SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );
	getDevice().SetRenderState( D3DRS_ZWRITEENABLE, TRUE );
}



// ------------------------------------------------------------------
//  terrain info and updating


void CTerrain::getNearestCell( float x, float y, int& cx, int& cy ) const
{
	assert( x >= 0.0f && x < gcon::WORLD_X || !"Coordinate X out of bounds" );
	assert( y >= 0.0f && y < gcon::WORLD_Y || !"Coordinate Y out of bounds" );
	
	x = x * CELL_INV_SIZE_X;
	y = y * CELL_INV_SIZE_Y;

	cx = math::round(x);
	cy = math::round(y);

	assert( cx >= 0 && cx < CELLS_X );
	assert( cy >= 0 && cy < CELLS_Y );
}

float CTerrain::getAltitude( float x, float y ) const
{
	assert( x >= 0.0f && x < gcon::WORLD_X || !"Coordinate X out of bounds" );
	assert( y >= 0.0f && y < gcon::WORLD_Y || !"Coordinate Y out of bounds" );

	x = x * CELL_INV_SIZE_X;
	y = y * CELL_INV_SIZE_Y;

	//
	// interpolate (2 cases: 2 triangles)

	//int ix = (int)x;
	//int iy = (int)y;
	int ix = math::round(x-0.5f);
	int iy = math::round(y-0.5f);
	float fx = x - (float)ix;
	float fy = y - (float)iy;

	float *alt = mAltitudes + iy*CELLS_X + ix;
	float z;

	if( fx > fy ) {
		// on 00, 01, 11 triangle
		float z00 = alt[0];
		float z01 = alt[1];
		float z11 = alt[CELLS_X+1];
		z = z00 + (z01-z00) * fx + (z11-z01) * fy;
	} else {
		// on 00, 11, 10 triangle
		float z00 = alt[0];
		float z10 = alt[CELLS_X];
		float z11 = alt[CELLS_X+1];
		z = z00 + (z11-z10) * fx + (z10-z00) * fy;
	}

	return z;
}

void CTerrain::beginUpdate()
{
}

void CTerrain::setAltitude( int idxX, int idxY, float z )
{
	mAltitudes[idxY*CELLS_X + idxX] = z;
}

void CTerrain::endUpdate()
{
	if( mVB )
		fillVB();
}


void CTerrain::getSlope( float x, float y, float& dx, float& dy ) const
{
	assert( x >= 0.0f && x < gcon::WORLD_X || !"Coordinate X out of bounds" );
	assert( y >= 0.0f && y < gcon::WORLD_Y || !"Coordinate Y out of bounds" );
	
	x = x * CELL_INV_SIZE_X;
	y = y * CELL_INV_SIZE_Y;
	
	int ix = math::round(x-0.5f);
	int iy = math::round(y-0.5f);
	
	float *alt = mAltitudes + iy*CELLS_X + ix;
	
	float z00 = alt[0];
	float z01 = alt[1];
	float z10 = alt[CELLS_X];

	dx = (z01 - z00) * CELL_INV_SIZE_X;
	dy = (z10 - z00) * CELL_INV_SIZE_Y;
}

