/**
 *  @file engine/Terrain.h
 *  Terrain class.
 */
#ifndef __TERRAIN_H
#define __TERRAIN_H

#include <d3d8.h>
#include <d3dx8math.h>
#include "DXObject.h"
#include "resource/ResourceId.h"


class CFrameTime;

/**
 *  World terrain.
 *
 *  Terrain is a heightfield that stretches over entire world on XY
 *  plane (so altitude is along Z axis).
 *
 *  Terrain can be queried for altitude at a specified point. Terrain
 *  altitudes may be updated. Terrain can render additional rectangular 
 *  areas above it (ex. for selection).
 *
 *  To access the terrain use CEngine.
 *  @see CEngine.
 */
class CTerrain : public CDXObject {
public:
	/// Terrain cells along X axis (must be a power of 2).
	enum { CELLS_X = 64 };
		/// Terrain cells along Y axis (must be a power of 2).
		enum { CELLS_Y = 64 };
	enum { CELLS = CELLS_X * CELLS_Y };
public:
	/**
	 *  Constructor.
	 *  @param tuTiles Texture U (X) tiles.
	 *  @param tvTiles Texture V (Y) tiles.
	 *  @param texture Texture resource ID.
	 */
	CTerrain( const CResourceId& effect, const CResourceId& texture, float tuTiles, float tvTiles );
	/// Destructor.
	virtual ~CTerrain();

	virtual HRESULT onCreateDevice( IDirect3DDevice8* device );
	virtual HRESULT onLostDevice();
	virtual HRESULT onResetDevice();
	virtual HRESULT onDestroyDevice();
	
	virtual void render( CFrameTime const& frameTiem );

	/// Sets "none" additional quad.
	void	clearQuad();
	/// Initializes additional quad.
	void	prepareQuad( float x1, float y1, float x2, float y2, float addZ );
	/// Renders quad (must be initialized first).
	void	renderQuad( D3DCOLOR color );

	/// Gets terrain altitude at XY coordinate.
	float	getAltitude( float x, float y ) const;
	/// Gets terrain altitude at XY cell index.
	float	getAltitudeIdx( int idxX, int idxY ) const { return mAltitudes[idxY*CELLS_X+idxX]; }
	/// Gets nearest cell index for XY coordinate.
	void	getNearestCell( float x, float y, int& cx, int& cy ) const;


	void getSlope( float x, float y, float& dx, float& dy ) const;

	
	/// Gets lowest altitude.
	float	getMinAltitude() const { return mMinAltitude; }
	/// Gets highest altitude.
	float	getMaxAltitude() const { return mMaxAltitude; }

	/// Gets terrain effect resource ID.
	const CResourceId& getEffect() const { return mEffect; }
	/// Sets terrain effect resource ID.
	void setEffect( const CResourceId& effect ) { mEffect = effect; }
	/// Gets terrain texture resource ID.
	const CResourceId& getTexture() const { return mTexture; }
	/// Sets terrain texture resource ID.
	void setTexture( const CResourceId& texture ) { mTexture = texture; }
	
	/**
	 *  Begin heightfield update.
	 *  @see setAltitude(), endUpdate().
	 */
	void	beginUpdate();
	/**
	 *  Update cell altitude.
	 *  All updates must happen between beginUpdate() and endUpdate().
	 *  @see beginUpdate(), endUpdate().
	 */
	void	setAltitude( int idxX, int idxY, float z );
	/**
	 *  Ends heightfield update.
	 *  @see beginUpdate(), setAltitude().
	 */
	void	endUpdate();

private:
	void	fillIB();
	void	fillVB();

private:
	float	mTuTiles, mTvTiles;
	float	mTuScale, mTvScale;
	CResourceId mEffect;
	CResourceId	mTexture;
	
	int		mVertexCount;
	int		mTriCount;
	IDirect3DIndexBuffer8*	mIB;
	IDirect3DIndexBuffer8*	mIBRect;
	IDirect3DVertexBuffer8*	mVB;
	IDirect3DVertexBuffer8*	mVBRect;
	int		mRectIndexCount;
	int		mRectVertexCount;
	
	float	mMinAltitude;
	float	mMaxAltitude;
	
	float*	mAltitudes;
};


#endif
