/**
 *  @file engine/console/Console.h
 *  Console stuff.
 */

#ifndef __CONSOLE_H
#define __CONSOLE_H

#include <windows.h>
#include <d3dx8math.h>
#include <string>
#include <list>

/// Console namespace.
namespace con
{
	class CChannel;
	class IConsoleRenderingContext;
	class IFilter;

	// --------------------------------------------------------------

	/**
	 *  A console.
	 *
	 *  Console enables arbitrary output via channels; filters can be used
	 *  to filter the output; and rendering contexts are for different
	 *  output rendering (eg. into file, into debug console, on screen,
	 *  etc.).
	 *
	 *  Basic use would be: get a channel: static getChannel(), and dump
	 *  stuff into it.
	 *
	 *  @see CChannel.
	 */
	class CConsole {
	public:
		static CConsole* const getConsole();
		/// Gets channel by name.
		static CChannel& getChannel( const std::string& name );
		virtual ~CConsole();

		virtual CChannel& openChannel( const std::string& name );
		virtual void closeChannel( CChannel& channel );

		void setDefaultFilter( IFilter& filter ) { mDefaultFilter = &filter; };
		IFilter* getDefaultFilter() const { return mDefaultFilter; };		

		void setDefaultRenderingContext( IConsoleRenderingContext& ctx );
		IConsoleRenderingContext* getDefaultRenderingContext() const { return mDefaultRenderingCtx; };
		
	protected:
		CConsole();
		CConsole( IConsoleRenderingContext& ctx );
		CConsole( IConsoleRenderingContext& ctx, IFilter& filter );

	private:
		typedef std::list<CChannel*>	TChannelList;
		IConsoleRenderingContext*	mDefaultRenderingCtx;
		IFilter*					mDefaultFilter;
		TChannelList				mChannels;

		static CConsole*			mSingleInstance;
	};

	// --------------------------------------------------------------

	/**
	 *  A console channel.
	 *  Has simple write() methods and some operator<< overloads.
	 *  @see CConsole.
	 */
	class CChannel {
	public:
		CChannel( CConsole& console, const std::string& name, IConsoleRenderingContext* renderingCtx = NULL, IFilter* filter = NULL );
		virtual ~CChannel();

		virtual void close();

		/// Writes to channel.
		virtual void write( const std::string& methodName, const std::string& message );
		/// Writes to channel.
		virtual void write( const std::string& message );

		void setRenderingContext( IConsoleRenderingContext& ctx ) { mRenderingCtx = &ctx; };
		IConsoleRenderingContext* getRenderingContext() const { return mRenderingCtx; };

		void setFilter( IFilter& filter ) { mFilter = &filter; };
		IFilter* getFilter() const { return mFilter; };

		CChannel& endl() { write( buf); buf.erase(); return *this;};
		CChannel& operator << (CChannel& (__cdecl * _f)(CChannel&)){  (*_f)(*this); return *this;};
		CChannel& operator << (const std::string& msg) { buf += msg.c_str(); return *this;}
		CChannel& operator << (char const* msg) { buf += msg; return *this; }
		CChannel& operator << (const int num) { char msg[33]; itoa( num, msg, 10); buf += msg; return *this;}
		CChannel& operator << (char c) { buf += c; return *this;}
		CChannel& operator << (const DWORD num) { char msg[33]; sprintf( msg, "%u", num); buf += msg; return *this;}
		CChannel& operator << (const float num) 
			{ char msg[33]; sprintf(msg, "%0.2f", num); buf += msg; return *this;}
		CChannel& operator << ( D3DXCOLOR const& col ) { char msg[66]; sprintf( msg, "(%01.2f %01.2f %01.2f %01.2f)", col.r, col.g, col.b, col.a ); buf += msg; return *this; }
		CChannel& operator << ( D3DXVECTOR3 const& col ) { char msg[66]; sprintf( msg, "(%01.2f %01.2f %01.2f)", col.x, col.y, col.z ); buf += msg; return *this; }
		CChannel& operator << ( D3DXVECTOR4 const& col ) { char msg[66]; sprintf( msg, "(%01.2f %01.2f %01.2f %01.2f)", col.x, col.y, col.z, col.w ); buf += msg; return *this; }
		CChannel& operator << ( D3DXQUATERNION const& col ) { char msg[66]; sprintf( msg, "(%01.2f %01.2f %01.2f %01.2f)", col.x, col.y, col.z, col.w ); buf += msg; return *this; }
		CChannel& operator << ( D3DXMATRIX const& m ) { char msg[128]; sprintf( msg,
			"(%01.2f %01.2f %01.2f %01.2f;\n"
			" %01.2f %01.2f %01.2f %01.2f;\n"
			" %01.2f %01.2f %01.2f %01.2f;\n"
			" %01.2f %01.2f %01.2f %01.2f)\n",
			m._11, m._12, m._13, m._14,
			m._21, m._22, m._23, m._24,
			m._31, m._32, m._33, m._34,
			m._41, m._42, m._43, m._44 ); buf += msg; return *this; }
		
		const std::string& getName() const { return mName; };

	protected:
		CConsole&		mParentConsole;
		std::string		buf;

	private:
		std::string					mName;
		IConsoleRenderingContext*	mRenderingCtx;
		IFilter*					mFilter;
	};
	
	inline CChannel& __cdecl flush(CChannel& cc) { return cc.endl(); }
	inline CChannel& __cdecl endl (CChannel& cc) { return cc.endl(); }

	// --------------------------------------------------------------
	
	class /*interface*/ IConsoleRenderingContext {
	public:
		virtual void write( const std::string& message ) = 0;
	};

	// --------------------------------------------------------------

	class /*interface*/ IFilter {
	public:
		virtual bool apply( const CChannel& channel, const std::string& message ) const = 0;
	};

} // namespace console

#endif
