#pragma warning(disable:4786)

#include "MyGame.h"
#include "../engine/EntityRenderer.h"
#include "../engine/Mesh.h"
#include "../engine/Terrain.h"
#include "../engine/TerrainGen.h"
#include "../engine/camera/BasicCamera.h"
#include "../engine/resource/PictureBundle.h"

// ENGINE CONSTANTS
#include "EngineConstants.h"
#include "../utils/Random.h"

#include "../engine/TerrainPicker.h"
#include "../engine/TextRenderer.h"
#include "../engine/Billboarder.h"
#include "TerrainModifier.h"
#include "../utils/Config.h"

#include <cassert>


const int TURN_TIME = (gcon::UPDATE_FREQ) * 40;

// ------------------------------------------------------------------
//  basic stuff


CMyGame::CMyGame( CConfig& config )
:	CBaseGame( config ),
	mTerrainCamera( NULL ),
	mPool( NULL ),
	mPoliticians( NULL )
{
}

CMyGame::~CMyGame()
{
	delete mPool;
	delete mPoliticians;
}

static void resetCamera( CBasicCamera& camera )
{
	camera.reset();
	//camera.setProjectionParams( 3.1415927f * 0.66f, 0.7f, 10.0f, 2000.0f );
	camera.getPosition() = D3DXVECTOR3( gcon::WORLD_X * 0.5f, gcon::WORLD_Y * 0.3f, 350.0f );
	camera.setBounded( true );
	camera.doPitch( -1.0f );
	camera.update( CFrameTime( 1.0f, 1.0f, 1, 0.1f ) );
	camera.stop();
}


CCamera* CMyGame::createCamera()
{
	mTerrainCamera = new CBasicCamera();
	return mTerrainCamera;
}



// ------------------------------------------------------------------
//  input


void CMyGame::onMouseLChange( bool pressed )
{
}

void CMyGame::onMouseRChange( bool pressed )
{
}

void CMyGame::onKeyChange( int key, bool pressed )
{
	if( pressed ) {
		switch( key ) {
		case '1': mActiveGoods = CGoods::MONEY; break;
		case '2': mActiveGoods = CGoods::JOBS; break;
		case '3': mActiveGoods = CGoods::GAMES; break;
		case '4': mActiveGoods = CGoods::FOOD; break;
		case ' ':
			if( isVictory() )
				restartGame();
			else if( mIntro )
				mIntro = false;
			else
				mTurnPause = false;
			break;
		}
	}
}

void CMyGame::onProcessInput( const CFrameTime& frameTime )
{
	assert( mTerrainCamera );

	float dt = frameTime.getDelta();
	if( getKeyboard().isKeyPressed('Z') )		mTerrainCamera->doYaw( 2.0f * dt );
	if( getKeyboard().isKeyPressed('C') )		mTerrainCamera->doYaw( -2.0f * dt );
	if( getKeyboard().isKeyPressed('S') )		mTerrainCamera->doMove( 1800.0f * dt );
	if( getKeyboard().isKeyPressed('X') )		mTerrainCamera->doMove( -1800.0f * dt );
	if( getKeyboard().isKeyPressed('A') )		mTerrainCamera->doStrafe( -1200.0f * dt );
	if( getKeyboard().isKeyPressed('D') )		mTerrainCamera->doStrafe( 1200.0f * dt );
}


// ------------------------------------------------------------------
//  initialization


void CMyGame::onInitialize()
{
	mIntro = true;
	mTotalTurns = getConfig().readI( "game", "turns", 10 );

	int n = getConfig().readI( "game", "politicians", 10 );
	mPoliticians = new TPolitPool( n*2 );
	
	mTerrainCamera->setTerrain( getTerrain() );

	int maxEntities = getEntities().getMaxEntities();
	int count = maxEntities/4;
	mPool = new TPool( count );

	n = getConfig().readI( "game", "promises", 100 );
	mPromises = new TPromisePool( n );

	CTerrainGen::fillTexture( getTerrain(), CResourceId("TerrainGray.png"), -50.0f, 100.0f );

	restartGame();
}


void CMyGame::restartGame()
{
	// remove all old
	TPolitPool::iterator pit;
	for( pit = mPoliticians->begin(); pit != mPoliticians->end(); ) {
		CPolitician& e = *pit;
		getEntities().remove( e );
		pit = mPoliticians->erase( pit );
	}
	TPromisePool::iterator prit;
	for( prit = mPromises->begin(); prit != mPromises->end(); ) {
		CPromise& e = *prit;
		getEntities().remove( e );
		prit = mPromises->erase( prit );
	}
	TPool::iterator it;
	for( it = mPool->begin(); it != mPool->end();  ) {
		CMyEntity& e = *it;
		getEntities().remove( e );
		it = mPool->erase( it );
	}

	int i;
	
	mJustChangedPolitician = false;
	mActiveGoods = CGoods::MONEY;
	
	mTurnNumber = 0;
	mEvilTurn = true;
	mTurnPause = true;
	mTurnCounter = 0;
	mChangePreference = CGoods::NOTHING;
	mFrame = 0;

	mScoreGood = mScoreEvil = 0;
	
	for( i = 0; i < CGoods::MAX_GOODS; ++i )
		mLeft[i] = 0;
	
	resetCamera( *mTerrainCamera );
	
	CPictureBundle& pb = CPictureBundle::getInstance();
	
	for( i = 0; i < mPool->capacity(); ++i ) {
		CMyEntity& e = mPool->add();
		getEntities().add( e );
		float x = random::randf(gcon::WORLD_X);
		float y = random::randf(gcon::WORLD_Y);
		float z = getTerrain().getAltitude( x, y ) + 5.0f;
		e.getPosition() = D3DXVECTOR3(x,y,z);
		e.getOldPosition() = e.getPosition();
		e.init();
		e.setPicture( *pb.getResourceById( *CGoods::getResource2( e.mPreference ) ) );
		e.setSize( 6.5f );
		e.setColor( 0xFFFFFFFF );
	}
	
	for( i = 0; i < mPoliticians->capacity(); ++i ) {
		CPolitician& e = mPoliticians->add( CPolitician(i<mPoliticians->capacity()/2?true:false) );
		getEntities().add( e );
		e.init();
		float x = random::randf(gcon::WORLD_X);
		float y = random::randf(gcon::WORLD_Y);
		float z = getTerrain().getAltitude( x, y ) + 15.0f;
		e.getPosition() = D3DXVECTOR3(x,y,z);
		e.getOldPosition() = e.getPosition();
		e.getVelocity() = D3DXVECTOR3( random::randfs(3.0f), random::randf(3.0f), 0.0f );
		e.setSize( 15.0f );
		e.setColor( 0xFFFFFFFF );
	}
}


// ------------------------------------------------------------------
//  game logic

int CMyGame::getNeededKills() const
{
	return mPool->capacity()/3;
}
bool CMyGame::isVictory() const
{
	return mPool->size() < mPool->capacity() - getNeededKills();
}

bool CMyGame::hasSpace() const
{
	return getEntities().getEntityCount() < getEntities().getMaxEntities();
}

void CMyGame::onUpdate()
{
	++mFrame;
	int rnd = rand() & 1023;
	if( rnd < 100 ) {
		mChangePreference = CGoods::NOTHING;
	}
	if( rnd > 1000 ) {
		mChangePreference = (CGoods::eGoods)(rnd%(CGoods::MAX_GOODS));
	}

	// clear grids
	mRay.beginFrame();
	mPoliticalGrid.beginFrame();

	if( mTurnPause ) {
		// update polits
		TPolitPool::iterator pit;
		for( pit = mPoliticians->begin(); pit != mPoliticians->end(); ++pit ) {
			CPolitician& e = *pit;
			e.getOldPosition() = e.getPosition();
		}
		// update promises
		TPromisePool::iterator prit;
		for( prit = mPromises->begin(); prit != mPromises->end(); ++prit ) {
			CPromise& e = *prit;
			e.getOldPosition() = e.getPosition();
		}
		// update entities
		TPool::iterator it;
		for( it = mPool->begin(); it != mPool->end(); ++it ) {
			CMyEntity& e = *it;
			e.getOldPosition() = e.getPosition();
		}
		return;
	}

	// change politician
	const CMouse& m = getMouse();
	if( !isVictory() && m.isMouseLPressed() ) {
		if( !mJustChangedPolitician ) {
			mRay.mHasRay = true;
			mRay.mOrigin = getCamera().getPosition();
			D3DXVECTOR3 dir = getCamera().getWorldRay( m.getMouseX(), m.getMouseY() );
			D3DXVec3Normalize( &mRay.mDirection, &dir );
		}
	} else {
		mJustChangedPolitician = false;
	}
	
	// update politicians
	TPolitPool::iterator pit;
	for( pit = mPoliticians->begin(); pit != mPoliticians->end(); ) {
		CPolitician& e = *pit;
		if( !mTurnNumber && !mTurnCounter && !e.isEvil() ) {
			// launch something to be fair - evils have first turn
			e.setGoods( (CGoods::eGoods)(rand()%(CGoods::MAX_GOODS-1)+1) );
			e.setCounter( 1 );
		}
		e.update();
		++pit;
	}
	
	// update promises
	TPromisePool::iterator prit;
	for( prit = mPromises->begin(); prit != mPromises->end(); ) {
		CPromise& e = *prit;
		e.update();
		if( e.getTTL() <= 0 ) {
			prit = mPromises->erase( prit );
			getEntities().remove( e );
		} else {
			++prit;
		}
	}

	// update entities
	TPool::iterator it;
	for( it = mPool->begin(); it != mPool->end();  ) {
		CMyEntity& e = *it;
		e.update();
		if( e.mFlags & CMyEntity::F_KILLED ) {
			it = mPool->erase( it );
			getEntities().remove( e );
		} else {
			++it;
		}
	}

	// turns
	++mTurnCounter;
	if( mTurnCounter >= TURN_TIME ) {
		mTurnCounter = 0;
		mEvilTurn = !mEvilTurn;
		++mTurnNumber;
		mTurnPause = true;
	}
}


// ------------------------------------------------------------------
//  rendering

void CMyGame::onRenderBeforeAll()
{
	CBillboarder& bb = getBillboarder();
	bb.beginBillboards();

	int n = mPoliticians->capacity()/2;

	// politicians
	TPolitPool::iterator pit;
	for( pit = mPoliticians->begin(); pit != mPoliticians->end(); ++pit ) {
		CPolitician& e = *pit;
		const CResourceId* rid = CGoods::getResource( e.getGoods() );
		if( rid ) {
			bb.putEntityBill( *rid, 0xC0FFFFFF, e, 20, 20 );
		}
	}

	if( isVictory() ) {
		if( mScoreEvil > mScoreGood ) {
			bb.putScreenBill( CResourceId("Politician1win.png"), 0xD0FFFFFF, -0.6f, -0.8f, 0.6f, 0.8f );
		} else if( mScoreGood > mScoreEvil ) {
			bb.putScreenBill( CResourceId("Politician2win.png"), 0xD0FFFFFF, -0.6f, -0.8f, 0.6f, 0.8f );
		} else {
			bb.putScreenBill( CResourceId("Politician1win.png"), 0xD0FFFFFF, -0.8f, -0.6f, -0.1f, 0.6f );
			bb.putScreenBill( CResourceId("Politician2win.png"), 0xD0FFFFFF, 0.1f, -0.6f, 0.8f, 0.6f );
		}
		bb.putScreenBill( CResourceId("PressSpace.png"), 0xF0FFFFFF, -0.4f, 0.71f, 0.4f, 0.93f );
	} else if( !mIntro ) {
		// indicators
		static CResourceId barid( "Bar.png" );

		if( !mTurnPause ) {
			float xx = ((float)mTurnCounter / TURN_TIME) * 2.0f - 1.0f;
			bb.putScreenBill( barid, 0xC0FFFF20, -1.0f, 0.95f, xx, 0.99f );
		}
		float yy1 = 0.99f - ((float)mScoreEvil / getNeededKills() ) * 2.0f;
		bb.putScreenBill( barid, 0xC0FF4040, -0.95f, yy1, -0.9f, 0.99f );
		float yy2 = 0.99f - ((float)mScoreGood / getNeededKills() ) * 2.0f;
		bb.putScreenBill( barid, 0xC040FF40, 0.9f, yy2, 0.95f, 0.99f );

		if( mEvilTurn ) {
			bb.putScreenBill( CResourceId("Politician1big.png"), 0xA0FFFFFF, -0.85f, 0.43f, -0.55f, 0.85f );
		} else {
			bb.putScreenBill( CResourceId("Politician2big.png"), 0xA0FFFFFF, 0.55f, 0.43f, 0.85f, 0.85f );
		}
		
		// active goods
		const CResourceId* rid = CGoods::getResource( mActiveGoods );
		if( rid ) {
			bb.putScreenBill( *rid, 0xD0FFFFFF, -0.14f, 0.69f, 0.14f, 0.99f );
		}

		if( mTurnPause ) {
			DWORD color = mEvilTurn ? 0xF0FFA0A0 : 0xF0A0FFA0;
			bb.putScreenBill( CResourceId("PressSpace.png"), color, -0.4f, -0.11f, 0.4f, 0.11f );
		}

		// left preferences
		for( int i = 1; i < CGoods::MAX_GOODS; ++i ) {
			float yy = -0.9f + i*0.2f;
			bb.putScreenBill( *CGoods::getResource2( (CGoods::eGoods)i ), 0xC0FFFFFF, 0.82f, yy-0.08f, 0.96f, yy+0.08f );
			float xx = 0.84f - ((float)mLeft[i] / mPool->capacity()) * 0.5f;
			bb.putScreenBill( barid, 0xC0FFFFFF, xx, yy-0.01f, 0.84f, yy+0.03f );
		}
	} else {
		bb.putScreenBill( CResourceId("PressSpace.png"), 0xF0FFFFFF, -0.4f, 0.71f, 0.4f, 0.93f );
	}

	
	bb.endBillboards();
}

void CMyGame::onRenderAfterWorld()
{
}

void CMyGame::onRenderAfterAll()
{
	char buf[100];

	if( isVictory() ) {
	} else if( !mIntro ) {
		sprintf( buf, "People: %i, Needed: %i", mPool->size(), getNeededKills() - (mPool->capacity()-mPool->size()) );
		getTextRenderer().renderScreenText( 4, 18, buf, 0x80FFFF00 );

		getTextRenderer().renderScreenText( 4, 32, "1234 - change active goods", 0x80FFFF00 );

		sprintf( buf, "The %s win by %i", mScoreEvil>mScoreGood?"reds":"greens", abs(mScoreEvil-mScoreGood) );

		getTextRenderer().renderScreenText( 10, 50, buf, mScoreEvil>mScoreGood ? 0xC0FF8080 : 0xC080FF80 );
	} else {
		getTextRenderer().renderScreenText( 20, 50,
			"GameJam'02 - Politics for the Masses by NeARAZ", 0xF0FF2020 );
		getTextRenderer().renderScreenText( 20, 70,
			"This is a two player turn-based game. The players play for red or green party.\n"
			"Each party has some politicians, and can offer/promise various things to the\n"
			"masses. You do this by selecting the thing with 1/2/3/4, and clicking on your\n"
			"politician.", 0xF0FFFFFF );
		getTextRenderer().renderScreenText( 20, 135,
			"When clicked, the politician starts spreading promises that roll down the hills\n"
			"and find the masses. If the dudes from the masses like the promise, they start\n"
			"seeking the politician that gave the promise. When they find it, they add a vote\n"
			"to his party and go to heaven.", 0xF0FFFFFF );
		getTextRenderer().renderScreenText( 20, 200,
			"You'll find that the masses like downhills, and the politicians like the high\n"
			"areas. Also at the very first move, the green party automatically launches some\n"
			"promises - that's to compensate for red's first move.", 0xF0FFFFFF );
		getTextRenderer().renderScreenText( 20, 250,
			"The controls are: 1/2/3/4 for switching the promises. S/X, A/D and Z/C for camera\n"
			"movement. Left mouse button for clicking. Space is needed between turns.", 0xF0FFFFFF );
	}
}

