/**
 * @file FixedPool.h
 * Fixed capacity pool container.
 */

#ifndef __FIXED_POOL_BY_VALUE_H__
#define __FIXED_POOL_BY_VALUE_H__


#include <vector>
#include <cassert>

//------------------------------------------------------------------------

/**
 * A pool container.
 *
 * Holds arbitrary type items by value. The good things about it:
 *	- O(1) addition complexity (no need to do "new"),
 *	- O(1) removal by iterator.
 *	- nearly cache-friendly - all items occupy a continuous area in memory, only the
 *	  iteration can be non-continuous.
 *  - items are never moved around, so you can easily have pointers to them, etc.
 *
 * Bad things:
 *	- Fixed maximum capacity,
 *	- All items must be equal in size (just like for arrays and std::vector),
 *	- Pool does not preserve the order. In detail: when some item is removed, the last
 *	  one gets into it's place (when iterating only - no memory copies are performed).
 *  - Iterating over a pool is somewhat more complex than over array or std::vector,
 *	  but generally cheaper than over a std::list (because of more cache friendliness).
 *	- Items must have proper copy constructor and default constuctor.
 *
 * Basic things about this pool are: size(), hasSpace(), clear(), add(), erase(),
 * begin() and end().
 *
 * @param T Type of the items.
 */
template<typename T>
class CFixedPool {
public:
	class iterator;
	class const_iterator;
private:
	typedef std::vector<T>		TDataVector;
	typedef std::vector<int>	TIndexVector;
private:
	TDataVector		mData;
	TIndexVector	mIndices;
	int				mSize;

public:

	/**
	 * Constructor.
	 * @param space Maximum capacity of the pool.
	 */
	explicit CFixedPool( int space ) : mSize(0) {
		assert( space >= 1 );
		mData.reserve( space );
		mIndices.reserve( space );
		for( int i = 0; i < space; i++ ) {
			mData.push_back( T() );
			mIndices.push_back( i );
		}
	}
	/// Destructor.
	~CFixedPool() {
		clear();
	};

	/// Returns current item count.
	int size() const { return mSize; };
	/// Returns total capacity.
	int capacity() const { return mData.size(); }
	/// Returns whether the pool is empty.
	bool empty() const { return mSize == 0; }
	/// Returns whether the pool is not full.
	bool hasSpace() const { return mSize < mData.size(); };

	/**
	 * Adds item to pool.
	 * Copies the given item to the pool, and returns a reference to a copy. You should
	 * use the returned reference to point to an item in the pool (not the given one!).
	 * @param value The item.
	 */
	T& add( const T& value = T() ) {
		assert( hasSpace() );
		int slot = mIndices[ mSize ];	// get empty slot index
		mData[slot] = value;			// fill it
		mSize++;						// grow
		return mData[slot];				// return the real object
	}

	/// Clears the pool.
	void clear();

	/**
	 * Erases a single item.
	 * @param it Iterator to the item that should be erased.
	 * @return Iterator to the next item (the supplied iterator gets invalid).
	 */
	iterator erase( iterator it );

public:
	friend class const_iterator;
	/**
	 * Const forward %iterator.
	 */
	class const_iterator {
		friend class iterator;
	public:
		const_iterator()
			{}
		const_iterator( const CFixedPool<T>& pool, int idx = 0 )
			: mPool(&pool), mIndex( idx ) { }
		const_iterator( const const_iterator& ci )
			: mPool(ci.mPool), mIndex(ci.mIndex) { }
		const_iterator( const iterator& ci )
			: mPool(ci.mPool), mIndex(ci.mIndex) { }
		const T& operator*() const {
			return mPool->mData[ mPool->mIndices[ mIndex ] ];
		}
		const T* operator->() const {
			return &**this;
		}
		const_iterator& operator++() {
			mIndex++;
			return *this;
		}
		const_iterator operator++(int) {
			const_iterator ti = *this;
			++*this;
			return ti;
		}
		bool operator==( const const_iterator& ci ) const {
			return mPool == ci.mPool && mIndex == ci.mIndex;
		}
		bool operator!=( const const_iterator& ci ) const {
			return !(*this == ci);
		}

	protected:
		const CFixedPool<T>*	mPool;
		int					mIndex;
	};

	friend class iterator;
	/**
	 * Forward %iterator.
	 */
	class iterator {
		friend class const_iterator;
	public:
		iterator()
			{}
		iterator( CFixedPool<T>& pool, int idx = 0 )
			: mPool(&pool), mIndex( idx ) { }
		iterator( const iterator& ci )
			: mPool(ci.mPool), mIndex(ci.mIndex) { }
		iterator( const const_iterator& ci )
			: mPool(ci.mPool), mIndex(ci.mIndex) { }
		T& operator*() const {
			int idx = mPool->mIndices[ mIndex ];
			return mPool->mData[ idx ];
		}
		T* operator->() const {
			return &**this;
		}
		iterator& operator++() {
			mIndex++;
			return *this;
		}
		iterator operator++(int) {
			iterator ti = *this;
			++*this;
			return ti;
		}
		bool operator==( const iterator& ci ) const {
			return mPool == ci.mPool && mIndex == ci.mIndex;
		}
		bool operator!=( const iterator& ci ) const {
			return !(*this == ci);
		}

		CFixedPool<T>& getPool() { return *mPool; }
		const CFixedPool<T>& getPool() const { return *mPool; }
		int& getIndex() { return mIndex; }
		const int& getIndex() const { return mIndex; }

		// after removal
		void revalidate() {
			if( mIndex > mPool->size() ) {
				mIndex = mPool->size();
			}
		}

	protected:
		CFixedPool<T>*	mPool;
		int			mIndex;
	};


	/// Returns %iterator to the first item.
	iterator begin() { return iterator( *this ); }
	/// Returns const %iterator to the first item.
	const_iterator begin() const { return const_iterator( *this ); }
	/// Returns %iterator to the item "beyond the last".
	iterator end() { return iterator( *this, size() ); }
	/// Returns const %iterator to the item "beyond the last".
	const_iterator end() const { return const_iterator( *this, size() ); }
};

//------------------------------------------------------------------------

template<class T>
void CFixedPool<T>::clear()
{
	mSize = 0;
}

template<class T>
CFixedPool<T>::iterator CFixedPool<T>::erase( iterator it )
{
	if( &it.getPool() != this )
		return it;
	int idx = it.getIndex();
	// remove and place last in place of it
	int i1 = idx;
	int i2 = mSize-1;
	// swap indices
	int t = mIndices[ i1 ];
	mIndices[ i1 ] = mIndices[ i2 ];
	mIndices[ i2 ] = t;
	mSize--;
	// revalidate iterator
	it.revalidate();
	return it;
}


#endif

