
#include "TerrainPicker.h"
#include "Terrain.h"
#include "../game/EngineConstants.h"
#include "../utils/MathUtil.h"
#include "Utils.h"
#include <cassert>


CTerrainPick CTerrainPicker::mNoPick = CTerrainPick( D3DXVECTOR3(0,0,0), -1, -1 );
static const float SMALL = 0.01f;
static const float SMALL_DELTA = 4.0f;


CTerrainPick CTerrainPicker::pick( const CTerrain& terrain, const D3DXVECTOR3& origin, const D3DXVECTOR3& dir )
{
	D3DXVECTOR3 or = origin;

	bool ok = utils::boundToWorld( or, dir, terrain.getMinAltitude(), terrain.getMaxAltitude() );
	if( !ok )
		return mNoPick;
	
	//
	// scale direction so that xy of it is slightly smaller than terrain cell

	D3DXVECTOR3 d = dir;
	float dirDX = fabsf( d.x );
	float dirDY = fabsf( d.y );
	float dirMax = dirDX > dirDY ? dirDX : dirDY;
	float cellK = (gcon::WORLD_X) / (CTerrain::CELLS_X) * 0.4f;
	if( dirMax > SMALL ) {
		d *= cellK / dirMax;
	}
	
	//
	// traverse until our line segment intersects the terrain

	float h1, h2;
	float dz1, dz2;
	D3DXVECTOR3 endWay = or;
	or -= d;
	h2 = terrain.getAltitude( endWay.x, endWay.y );
	do {
		or = endWay;
		endWay += d;
		if( endWay.x < 0.0f || endWay.x >= gcon::WORLD_X )
			return mNoPick;
		if( endWay.y < 0.0f || endWay.y >= gcon::WORLD_Y )
			return mNoPick;
		h1 = h2;
		h2 = terrain.getAltitude( endWay.x, endWay.y );
		dz1 = h1 - or.z;
		dz2 = h2 - endWay.z;
	} while( dz1 * dz2 > 0.0f );

	//
	// find intersection by division by 2 method

	float delta;
	D3DXVECTOR3 halfWay;
	do {
		halfWay = (or + endWay) * 0.5f;
		float h = terrain.getAltitude( halfWay.x, halfWay.y );
		delta = h - halfWay.z;
		dz1 = h1 - or.z;
		dz2 = h2 - endWay.z;
		assert( dz1 * dz2 <= 0.0f );
		if( dz1 * delta >= 0.0f ) {
			or = halfWay;
			h1 = h;
		} else {
			endWay = halfWay;
			h2 = h;
		}
	} while( fabsf(delta) > SMALL_DELTA );

	int cellX, cellY;
	terrain.getNearestCell( halfWay.x, halfWay.y, cellX, cellY );
	
	return CTerrainPick( halfWay, cellX, cellY );
}
