/**
 *  @file engine/EntityRayPicker.h
 *  Utility to check entity-ray intersection.
 */
#ifndef __ENTITY_RAY_PICKER_H
#define __ENTITY_RAY_PICKER_H

#include <d3dx8math.h>
#include <cassert>

#include "Grid.h"
#include "Terrain.h"
#include "Entity.h"
#include "Utils.h"
#include "../game/EngineConstants.h"


struct SSectorWithRay {
public:
	SSectorWithRay() : mHasRay( false ) { }

	void	beginFrame() { mHasRay = false; }
	void	endFrame() { }

public:
	bool		mHasRay;
	D3DXVECTOR3	mOrigin;
	D3DXVECTOR3	mDirection;
};

static const float RAYPICK_SMALL_DIR = 0.1f;


/**
 *  Utility to check entity-ray intersection.
 *  Grid sector class must have the following public members:
 *		- bool mHasRay;
 *		- D3DXVECTOR3 mOrigin;
 *		- D3DXVECTOR3 mDirection;
 *
 *  @param GRID Grid class.
 */
template<typename GRID>
class CEntityRayPicker {
public:
	/**
	 *  Puts ray into grid.
	 *  For all sectors that are touched by a ray, puts the ray into
	 *  the sector. Does not modify untouched sectors.
	 *  @param grid The grid.
	 *  @param origin Origin of the ray.
	 *  @param dir Direction of the ray (can be not unit vector).
	 */
	static void putRay( GRID& grid, const D3DXVECTOR3& origin, const D3DXVECTOR3& dir )
	{
		D3DXVECTOR3 or = origin;
		D3DXVECTOR3 normDir;
		D3DXVec3Normalize( &normDir, &dir );
		
		bool ok = utils::boundToWorld( or, dir, -1.0e6f, 1.0e6f );
		if( !ok )
			return;
		
		float dirDX = fabsf(dir.x);
		float dirDY = fabsf(dir.y);
		if( dirDX < RAYPICK_SMALL_DIR && dirDY < RAYPICK_SMALL_DIR ) {
			GRID::TSector& sector = grid.getSectorByPosition( or.x, or.y );
			sector.mHasRay = true;
			sector.mOrigin = or;
			sector.mDirection = dir;
			return;
		}
		
		float k = ( dirDX > dirDY ) ? (grid.getStepX() / dirDX) : (grid.getStepY() / dirDY);
		D3DXVECTOR3 d = dir * k * 0.15f;
		
		int oldSector = -1;
		do {
			int sector = grid.position2Index( or.x, or.y );
			if( sector != oldSector ) {
				GRID::TSector& s = grid.getSectorByIndex( sector );
				s.mHasRay = true;
				s.mOrigin = or;
				s.mDirection = normDir;
				oldSector = sector;
			}
			or += d;
		} while( or.x >= 0.0f && or.x < (gcon::WORLD_X) && or.y >= 0.0f && or.y < (gcon::WORLD_Y) );
	}
	
	/**
	 *  Checks entity-ray intersection.
	 *  Checks intersection of an entity with a ray that is in a given
	 *  sector. If a sector does not contain a ray, no intersection is reported.
	 *  @param e Entity.
	 *  @param sector Sector.
	 *  @return True if intersection occured, false otherwise.
	 */
	static bool intersects( const CEntity& e, const GRID::TSector& sector )
	{
		if( !sector.mHasRay )
			return false;
		// actually a ray-sphere intersection
		D3DXVECTOR3 diff = e.getPosition() - sector.mOrigin;
		float sqrLen = D3DXVec3LengthSq( &diff );
		float t = D3DXVec3Dot( &diff, &sector.mDirection );
		if( t > 0 )
			diff -= sector.mDirection * t;
		
		sqrLen = D3DXVec3LengthSq( &diff );
		if( sqrLen > e.getSize() * e.getSize() * 2.2f )
			return false;
		
		return true;
	}
};



#endif
