/**
 *  @file engine/camera/BasicCamera.h
 *  Basic camera.
 */

#ifndef __BASIC_CAMERA_H
#define __BASIC_CAMERA_H

#include "Camera.h"


class CTerrain;

/**
 *  Basic camera.
 *  A camera that can move, strafe, rise; yaw, pitch and roll. Also has controllable
 *  (via damping) inertia. Can be local-relative or world-relative. Can respond
 *  to gravity and/or don't allow to fly and/or don't allow to go outside the world.
 */
class CBasicCamera : public CCamera {
public:
	CBasicCamera();
	
	const D3DXVECTOR3& getPosition() const { return mPosition; }
	D3DXVECTOR3& getPosition() { return mPosition; }

	/** Moves along "front" axis. */
	void	doMove( float distance ) { mDistMove += distance; }
	/** Moves along "right" axis. */
	void	doStrafe( float distance ) { mDistStrafe += distance; }
	/** Moves along "up" axis. */
	void	doRise( float distance ) { mDistRise += distance; }

	/** Turns horizontally (around "up" axis). */
	void	doYaw( float radians ) { mTurnYaw += radians; }
	/** Turns vertically (around "right" axis). */
	void	doPitch( float radians ) { mTurnPitch += radians; }
	/** Turns around "front" axis. */
	void	doRoll( float radians ) { mTurnRoll += radians; }

	void	stop();

	virtual void update( CFrameTime const& frameTime );

	bool	isWorldRelative() const { return mWorldRelative; }
	void	setWorldRelative( bool wr ) { mWorldRelative = wr; }
	bool	isGravitable() const { return mGravitable; }
	void	setGravitable( bool g ) { mGravitable = g; }
	bool	isWalkOnly() const { return mWalkOnly; }
	void	setWalkOnly( bool wo ) { mWalkOnly = wo; }
	bool	isBounded() const { return mBounded; }
	void	setBounded( bool b ) { mBounded = b; }

	float	getMoveDamping() const { return mMoveDamping; }
	void	setMoveDamping( float md ) { mMoveDamping = md; }
	float	getTurnDamping() const { return mTurnDamping; }
	void	setTurnDamping( float td ) { mTurnDamping = td; }
	float	getMoveMax() const { return mMoveMax; }
	void	setMoveMax( float mm ) { mMoveMax = mm; }
	float	getTurnMax() const { return mTurnMax; }
	void	setTurnMax( float tm ) { mTurnMax = tm; }

	float	getMinAltitude() const { return mMinAltitude; }
	void	setMinAltitude( float m ) { mMinAltitude = m; }

	void	setTerrain( CTerrain& terrain ) { mTerrain = &terrain; }
	CTerrain const* getTerrain() const { mTerrain; }
	CTerrain* getTerrain() { mTerrain; }
	
protected:
	static float clamp( float v, float vmin, float vmax );
	void	limitMotion();
	
private:
	D3DXVECTOR3		mPosition;
	D3DXVECTOR3		mFront;
	D3DXVECTOR3		mUp;
	D3DXVECTOR3		mRight;
	float		mDistMove;
	float		mDistStrafe;
	float		mDistRise;
	float		mTurnYaw;
	float		mTurnPitch;
	float		mTurnRoll;

	float		mMoveDamping;
	float		mTurnDamping;

	float		mMoveMax;
	float		mTurnMax;

	bool		mWorldRelative;
	bool		mGravitable;
	bool		mWalkOnly;
	bool		mBounded;

	float		mMinAltitude;

	CTerrain*	mTerrain;
};

#endif
