#include "BasicCamera.h"

#include <cassert>
#include "../Terrain.h"
#include "../FrameTime.h"
#include "../../utils/MathUtil.h"

// ENGINE CONSTANTS
#include "../../game/EngineConstants.h"


CBasicCamera::CBasicCamera()
:	mPosition(0,0,0), mFront(0,1,0), mUp(0,0,1), mRight(1,0,0),
	mDistMove(0), mDistStrafe(0), mDistRise(0),
	mTurnYaw(0), mTurnPitch(0), mTurnRoll(0),
	mMoveDamping( 2.0f ), mTurnDamping( 2.0f ),
	mMoveMax( 150.0f ), mTurnMax( 3.0f ),
	mWorldRelative(true), mGravitable(false),
	mWalkOnly(false), mBounded(false),
	mMinAltitude(20.0f), mMaxAltitude(200000.0f),
	mTerrain( NULL )
{
}


void CBasicCamera::stop()
{
	mDistMove = mDistStrafe = mDistRise = 0.0f;
	mTurnYaw = mTurnPitch = mTurnRoll = 0.0f;
}

float CBasicCamera::clamp( float v, float vmin, float vmax )
{
	if( v < vmin )
		return vmin;
	if( v > vmax )
		return vmax;
	return v;
}

void CBasicCamera::limitMotion()
{
	mDistMove = clamp( mDistMove, -mMoveMax, mMoveMax );
	mDistStrafe = clamp( mDistStrafe, -mMoveMax, mMoveMax );
	mDistRise = clamp( mDistRise, -mMoveMax, mMoveMax );
	mTurnYaw = clamp( mTurnYaw, -mTurnMax, mTurnMax );
	mTurnPitch = clamp( mTurnPitch, -mTurnMax, mTurnMax );
	mTurnRoll = clamp( mTurnRoll, -mTurnMax, mTurnMax );
}

void CBasicCamera::update( CFrameTime const& frameTime )
{
	assert( mTerrain );

	limitMotion();

	//
	// rotate
	
	D3DXMATRIX matYaw;
	D3DXMatrixRotationAxis( &matYaw, mWorldRelative ? &D3DXVECTOR3(0,0,1) : &mUp, mTurnYaw * frameTime.getDelta() );
	D3DXMATRIX matPitch;
	D3DXMatrixRotationAxis( &matPitch, &mRight, mTurnPitch * frameTime.getDelta() );
	D3DXMATRIX matRoll;
	D3DXMatrixRotationAxis( &matRoll, &mFront, mTurnRoll * frameTime.getDelta() );
	D3DXMATRIX m = matYaw * matPitch * matRoll;

	D3DXVec3TransformCoord( &mFront, &mFront, &m );
	D3DXVec3TransformCoord( &mRight, &mRight, &m );
	D3DXVec3TransformCoord( &mUp, &mUp, &m );

	//
	// move

	D3DXVECTOR3 vf = mFront;
	D3DXVECTOR3 vr = mRight;
	D3DXVECTOR3 vu = mUp;
	if( mWorldRelative ) {
		vf.z = 0.0f; D3DXVec3Normalize( &vf, &vf );
		vr.z = 0.0f; D3DXVec3Normalize( &vr, &vr );
		vu.x = 0.0f; vu.y = 0.0f; vu.z = 1.0f;
	}

	mPosition += vf * (mDistMove * frameTime.getDelta());
	mPosition += vr * (mDistStrafe * frameTime.getDelta());
	mPosition += vu * (mDistRise * frameTime.getDelta());

	//
	// gravity?

	if( mGravitable )
		mPosition.z -= gcon::GRAVITY * frameTime.getDelta();

	//
	// bounds? terrain? walkonly?

	float xx = mPosition.x;
	float yy = mPosition.y;
	if( xx < 0.0f )
		xx = 0.0f;
	else if( xx >= gcon::WORLD_X )
		xx = gcon::WORLD_X - gcon::EPSILON;
	if( yy < 0.0f )
		yy = 0.0f;
	else if( yy >= gcon::WORLD_Y )
		yy = gcon::WORLD_Y - gcon::EPSILON;

	float zz = mTerrain->getAltitude( xx, yy );

	if( mWalkOnly ) {
		mPosition.z = zz + mMinAltitude;
	} else {
		if( mPosition.z < zz + mMinAltitude )
			mPosition.z = zz + mMinAltitude;

		if( mPosition.z > mMaxAltitude )
			mPosition.z = mMaxAltitude - gcon::EPSILON;
	}

	if( mBounded ) {
		mPosition.x = xx;
		mPosition.y = yy;
	}


	//
	// damp
	
	float moveDamp = 1.0f / (1.0f + mMoveDamping * frameTime.getDelta());
	float turnDamp = 1.0f / (1.0f + mTurnDamping * frameTime.getDelta());
	mDistMove *= moveDamp;
	mDistStrafe *= moveDamp;
	mDistRise *= moveDamp;
	mTurnYaw *= turnDamp;
	mTurnPitch *= turnDamp;
	mTurnRoll *= turnDamp;


	//
	// assemble matrices

	math::buildMatrix( mCameraMatrix, mRight, mUp, mFront, mPosition );
	updateMatrices();
}
