#include "GodzillaManager.h"
#include "../engine/Mesh.h"
#include "../engine/Terrain.h"
#include "../engine/FrameTime.h"

// ENGINE CONSTANTS
#include "EngineConstants.h"
#include "../utils/Random.h"

#include "MyGame.h"

static const float ANIM_SPEED = 3.0f;
static const float DIR_KEEP_DURATION = 10.0f;

D3DXVECTOR3 CGodzilla::genDirection()
{
	D3DXVECTOR3 dir;
	D3DXVec3Normalize( &dir, &random::randv3s(1) );
	return dir;
}

CGodzilla::CGodzilla()
{
	mCurrentMesh = NULL;
	mLastDirChange = 0.0f;
	mPosition = D3DXVECTOR3(0,0,0);
	mOldPosition = D3DXVECTOR3(0,0,0);
	mStartTime = 0;
	mSpeed = 1.0f;
	mScale = 1.0f;
	mFlying = false;

	mDirection = genDirection();

	mAnimator.add( mCurrentMesh =  new CMesh(
		CResourceId("step1.x"),
		CResourceId("TexturedMesh.sha"),
		CResourceId("godzilla_texture.png") ) );
	mAnimator.add( new CMesh(
		CResourceId("step2.x"),
		CResourceId("TexturedMesh.sha"),
		CResourceId("godzilla_texture.png") ) );
	mAnimator.add( new CMesh(
		CResourceId("step3.x"),
		CResourceId("TexturedMesh.sha"),
		CResourceId("godzilla_texture.png") ) );
	mAnimator.add( new CMesh(
		CResourceId("step4.x"),
		CResourceId("TexturedMesh.sha"),
		CResourceId("godzilla_texture.png") ) );
}

void CGodzilla::reset( CFrameTime const& ft )
{
	mStartTime = ft.getTime();
}

void CGodzilla::animation( CFrameTime const& ft )
{
	mAnimator.update( ft );

	CMesh* mesh = mAnimator.getValue( mStartTime, ANIM_SPEED );

	if( mesh != mCurrentMesh )
	{
		if( mCurrentMesh ) {
			CMyGame::getInstance().getObjects().removeObject( *mCurrentMesh );
		}
		CMyGame::getInstance().getObjects().addObject( *mesh );
		mCurrentMesh = mesh;
	}
}

void CGodzilla::movement( CFrameTime const& ft )
{
	// generate new movement direction
	if( ft.getTime() - mLastDirChange >= DIR_KEEP_DURATION )
	{
		mDirection = genDirection();
		mLastDirChange = ft.getTime();
	}

	D3DXVECTOR3 pos = mPosition;
	pos += mDirection * mSpeed * ft.getDelta();

	// bound
	if( pos.x >= gcon::WORLD_X || pos.x < 0.0f ) {
		mDirection = genDirection();
		mLastDirChange = ft.getTime();

		pos.x = mOldPosition.x;
	}
	if( pos.y >= gcon::WORLD_Y || pos.y < 0.0f ) {
		mDirection = genDirection();
		mLastDirChange = ft.getTime();

		pos.y = mOldPosition.y;
	}

	pos.z -= 10;

	// keep on the terrain
	float alt = CMyGame::getInstance().getTerrain().getAltitude( pos.x, pos.y ) + CGodzilla::HEIGHT + 15;

	if( pos.z < alt )
		pos.z = alt;

	mOldPosition = mPosition;

	mPosition = pos;
}

void CGodzilla::update()
{
	D3DXMATRIX tm, sm, rm, rm1, orm;
	
	if( mCurrentMesh )
	{
		D3DXMatrixIdentity( &orm );

		orm._11 = mDirection.x; orm._12 = mDirection.y;
		orm._21 = -mDirection.y; orm._22 = mDirection.x;
		orm._33 = orm._44 = 1.0f;
		
		D3DXMatrixRotationX( &rm, D3DX_PI/2.0f );
		D3DXMatrixRotationZ( &rm1, D3DX_PI/2.0f );
		D3DXMatrixTranslation( &tm, mPosition.x, mPosition.y, mPosition.z );
		D3DXMatrixScaling( &sm, mScale, mScale, mScale );

		mCurrentMesh->getWorldMatrix() = rm*rm1*orm*sm*tm;
	}
}


//
//
//
CGodzillaManger::CGodzillaManger()
{
}

void CGodzillaManger::update( CFrameTime const& ft )
{
	for( TGodzillaContainer::iterator it = mGodzillas.begin(); it != mGodzillas.end(); ++it )
	{
		(*it)->animation( ft );

		if( !(*it)->flying() ) {
			(*it)->movement( ft );
			put( CMyGame::getInstance().mGrid, **it );
		}

		(*it)->update();
	}
}

void CGodzillaManger::put( TGrid& grid, CGodzilla& godz )
{
	CMyGame::TGrid::TSector& sec = grid.getSectorByPosition( godz.position().x, godz.position().y );
	sec.mGodzillaFactor += CGodzilla::KILL_FACTOR;
}
