#pragma warning(disable:4786)

#include "MyGame.h"
#include "../engine/EntityRenderer.h"
#include "../engine/Mesh.h"
#include "../engine/Terrain.h"
#include "../engine/TerrainGen.h"
#include "../engine/Billboarder.h"
#include "../engine/camera/BasicCamera.h"
#include "../engine/TextRenderer.h"
#include "../engine/resource/PictureBundle.h"
#include "../utils/Random.h"
#include "../utils/Config.h"

#include "EngineConstants.h"

#include <cassert>
#include <algorithm>
#include <functional>

#include "myentity.h"
#include "deciders.h"

// ------------------------------------------------------------------
//  basic stuff

CMyGame::CMyGame( CConfig& config )
:	CBaseGame( config ),
	mTerrainCamera( NULL ),
	mPool( NULL ),
	mKilled( 0 ),
	mDraging( false )
{
}

CMyGame::~CMyGame()
{
	delete mPool;
}

CCamera* CMyGame::createCamera()
{
	mTerrainCamera = new CBasicCamera();
	mTerrainCamera->getPosition() = D3DXVECTOR3( gcon::WORLD_X * 0.5f, gcon::WORLD_Y * 0.5f, 100.0f );

	return mTerrainCamera;
}

// ------------------------------------------------------------------
//  input


void CMyGame::onMouseLChange( bool pressed )
{
	if( pressed )
	{
		mShotDir = mTerrainCamera->getWorldRay( getMouse().getMouseX(), getMouse().getMouseY() );
		mHaveShot = true;
		mNoShot = false;
	}
}

void CMyGame::onMouseRChange( bool pressed )
{
	if( !mDraging ) {
		mDragStartX = getMouse().getMouseX();
		mDragStartY = getMouse().getMouseY();
	}
	mDraging = pressed;
}

void CMyGame::onKeyChange( int key, bool pressed )
{
}

void CMyGame::onProcessInput()
{
	static float oldX, oldY; 

	switch( mState )
	{
	case PRESS:
		if( getKeyboard().isKeyPressed(VK_SPACE) )
			mState = GAME;

		oldX = getMouse().getMouseX();
		oldY = getMouse().getMouseY();

		break;

	case GAME:
	{
		//D3DXVECTOR3 oldpos = mTerrainCamera->getPosition();

		if( getKeyboard().isKeyPressed(VK_UP) )		mTerrainCamera->doPitch( 0.05f );
		if( getKeyboard().isKeyPressed(VK_DOWN) )	mTerrainCamera->doPitch( -0.05f );
		if( getKeyboard().isKeyPressed(VK_LEFT) )	mTerrainCamera->doYaw( 0.05f );
		if( getKeyboard().isKeyPressed(VK_RIGHT) )	mTerrainCamera->doYaw( -0.05f );
		if( getKeyboard().isKeyPressed('S') )		mTerrainCamera->doMove( 70.0f );
		if( getKeyboard().isKeyPressed('X') )		mTerrainCamera->doMove( -70.0f );
		if( getKeyboard().isKeyPressed('Z') )		mTerrainCamera->doStrafe( -40.0f );
		if( getKeyboard().isKeyPressed('C') )		mTerrainCamera->doStrafe( 40.0f );
		//if( getKeyboard().isKeyPressed('Q') )		mTerrainCamera->doRise( 40.0f );
		//if( getKeyboard().isKeyPressed('A') )		mTerrainCamera->doRise( -40.0f );

		break;
	}

	case GAMEOVER:
		break;
	}
}


// ------------------------------------------------------------------
//  initialization


void CMyGame::onInitialize()
{
	mTerrainCamera->setTerrain( getTerrain() );
	mTerrainCamera->setBounded( true );
	mTerrainCamera->setMaxAltitude( 400 );
	mTerrainCamera->setWalkOnly( true );
	mTerrainCamera->setTurnDamping( 50 );

	int maxEntities = getEntities().getMaxEntities();
	mPool = new TPool( maxEntities );

	CPictureBundle& pb = CPictureBundle::getInstance();
	const CPicture& pic1 = *pb.getResourceById( CResourceId("guy.png") );

	CTerrainGen::fillTexture( getTerrain(), CResourceId("sand_trn_bw.png"), -50.0f, 100.0f );
//	CTerrainGen::fillTexture( getTerrain(), CResourceId("TerrainGray.png"), -50.0f, 100.0f );

	mSuckDecider = new CSuckDecider();

	int count = maxEntities;
	mTotalAtStart = maxEntities;
	mCanKill = maxEntities*0.5f;
	int const ufo_count = getConfig().readI( "game", "ufoCount", 8 );
	int i;

	for( i=0; i<count; ++i )
	{
		CMyEntity& e = mPool->add();
		getEntities().add( e );

		e.init();

		e.getPosition() = D3DXVECTOR3( random::randf(gcon::WORLD_X), random::randf(gcon::WORLD_Y), 0.0f );
		e.mVelocity = CMyEntity::genWalkVelocity();
		e.getOldPosition() = e.getPosition();
		e.mDeciderCounter = 10;
		e.mDecider = mSuckDecider;
		e.mFlags = CMyEntity::F_GRAVITY;//|CMyEntity::F_WALKING;
		e.setSize( 4.0f );
		e.setColor( 0xFFFFFFFF );

		e.setPicture( pic1 );
	}

	for( i = 0; i<ufo_count; ++i )
	{
		CMesh* ufo_mesh = new CMesh( CResourceId("ufo.x"), CResourceId("UfoMesh.sha"), CResourceId("ufo.png") );
		assert( ufo_mesh );

		CUfo* ufo = new CUfo(
			D3DXVECTOR3( random::randf( gcon::WORLD_X/10 ), random::randf( gcon::WORLD_Y/10 ), 100 ),
			*ufo_mesh );

//		ufo->mVelocity = random::randv3s( 4.0f );
		ufo->mSuckPoint = CUfo::genTerrainPoint();
		//ufo->speed( 100.0f );
		ufo->speed( 500.0f );
		ufo->scale( 25 );
		
		ufo->update();

		getObjects().addObject( *ufo_mesh );
		mUfoManager.mUfos.push_back( ufo );
	}

	for( i=0; i<mGrid.getSecsCount(); ++i ) {
		SGodzillaUfoSector& sec = mGrid.getSectorByIndex( i );
		sec.mGodzillaFactor = 0;
		sec.mUfoFactor = 0;
		sec.mUfo = NULL;
		sec.mUfoHit = false;
	}

	CUfo::SUCK_COUNT = getConfig().readI( "game", "suckCount", 200 );
	CUfo::HIT_DAMAGE = getConfig().readI( "game", "hitDamage", 35 );
	int init_godz = getConfig().readI( "game", "initGodz", 1 );

	for( i=0; i<init_godz; ++i )
	{
		CGodzilla* godz = new CGodzilla;
		godz->getOldPosition() = godz->position( D3DXVECTOR3(100,100,100) );
		godz->speed( 30 );
		godz->scale( 2 );
		mGodzillaManager.add( *godz );
	}

	mState = PRESS;
	mStartTime = 0;
	mNoShot = false;
}


// ------------------------------------------------------------------
//  game logic


void CMyGame::onUpdate()
{
	// update entities
	TPool::iterator it;
	for( it = mPool->begin(); it != mPool->end();  ) {
		CMyEntity& e = *it;
		assert( e.mDeciderCounter >= 0 );
		e.update();
		if( e.mFlags & CMyEntity::F_KILLED ) {
			it = mPool->erase( it );
			getEntities().remove( e );
			++mKilled;
		} else {
			++it;
		}
	}

	switch( mState )
	{
	case PRESS:
		break;

	case GAME:
	{
		if( mKilled >= mCanKill ) {
			mState = GAMEOVER;
		}
		break;
	}

	case GAMEOVER:
		break;
	}
}


// ------------------------------------------------------------------
//  rendering

void CMyGame::onRenderBeforeAll( CFrameTime const& ft )
{
	switch( mState )
	{
	case PRESS:
		mStartTime = ft.getTime();
		break;

	case GAME:
		if( mDraging )
		{
			float p = (mDragStartX - getMouse().getMouseX()) * 100;
			float y = (mDragStartY - getMouse().getMouseY()) * 100;

			mDragStartX = getMouse().getMouseX();
			mDragStartY = getMouse().getMouseY();

			mTerrainCamera->setYaw( p );
			mTerrainCamera->setPitch( y );
		}

		if( mHaveShot && !mNoShot ) {
			mUfoManager.shoot( ft, mTerrainCamera->getPosition(), mShotDir );
			mHaveShot = false;
			mNoShot = true;
		}
		mUfoManager.update( ft );
		mGodzillaManager.update( ft );

		if( ft.getTime() - mStartTime >= GAME_LENGTH )
			mState = WON;
		break;

	case WON:
		break;

	case GAMEOVER:
		break;
	}
}

void CMyGame::onRenderAfterWorld( CFrameTime const& ft )
{
}

char* instr=
"Control: Z,C,S,X - move; mouse+right-button - direction; mouse+Left-button - fire\n"
"\n"
"Your task is to save at least half the population you started with (that's the progress bar for)\n"
"in two minutes. By clicking the mouse you can shoot the UFO, though you can damage it only when\n"
"it sucks humans. After three successfull hits, UFO crashes (to respawn). If it sucks enough humans,\n"
"it goes to the outerworld to transform it to something wild. Don't panic, if you won't be able to\n"
"hit some UFO :)\n"
"\n"
"Big thanks to Gron for making the models\n"
"simple, 2002 08 26";

void CMyGame::onRenderAfterAll( CFrameTime const& ft )
{
	/*
	char buf[1024];
	sprintf( buf,
		"Entities: %d\n"
		"Ufo[0] state: %d, sucked=%d, trans=%d\n"
		"Ufo[1] state: %d, sucked=%d, trans=%d",
		getEntities().getEntityCount(),
		mUfoManager.mUfos[0]->state(), mUfoManager.mUfos[0]->sucked(), mUfoManager.mUfos[0]->needTransform(),
		mUfoManager.mUfos[1]->state(), mUfoManager.mUfos[1]->sucked(), mUfoManager.mUfos[1]->needTransform() );
	getTextRenderer().renderScreenText( 0,16, buf );
	*/

	getBillboarder().beginBillboards();

	switch( mState )
	{
	case PRESS:
		{
		getBillboarder().putScreenBill( CResourceId("start_pic.png"),
			0xffffffff,
			-0.5f, -0.7f, 0.5f, -0.2f );
		getTextRenderer().renderScreenText( 10, 400, instr, 0xff000000 );
		break;
		}
	case GAME:
		{

		for( CUfoManager::THitContainer::iterator it = mUfoManager.hits().begin(); it != mUfoManager.hits().end(); )
		{
			if( ft.getTime() - it->time >= 0.5 ) {
				it = mUfoManager.hits().erase( it );
				continue;
			}

			// show
			D3DXVECTOR3 rp = random::randv3s( 10 );
			getBillboarder().putWorldBill( it->tex, 0xffffffff, it->pos + rp, 64, 64 );
			++it;
		}

		float k = 1.0f - float(mKilled) / float(mCanKill);
		getBillboarder().putScreenBill( CResourceId("bar.png"),
			0xffff0000,
			-0.990f, 0.950f, -0.990f + 2*0.990f*k, 0.990f );

		int height = getConfig().readI( "screen", "height", 480 );

		// show time
		char buf[256];
		int time_left = GAME_LENGTH - (ft.getTime() - mStartTime);
		sprintf( buf, "%02d:%02d\n%d", int(time_left / 60), int(time_left%60), getEntities().getEntityCount() );
		getTextRenderer().renderScreenText( 8, height-40, buf, 0xffffffff );

		break;
		}

	case WON:
		{
		getBillboarder().putScreenBill( CResourceId("game_won.png"),
			0xffffffff,
			-0.5f, -0.25f, 0.5f, 0.25f );
		break;
		}

	case GAMEOVER:
		{
		getBillboarder().putScreenBill( CResourceId("game_over.png"),
			0xffffffff,
			-0.5f, -0.25f, 0.5f, 0.25f );
		break;
		}
	}

	getBillboarder().endBillboards();
}
