#include "UfoManager.h"
#include "MyGame.h"
#include "../engine/Mesh.h"
#include "../engine/Terrain.h"
#include "../engine/Grid.h"

// ENGINE CONSTANTS
#include "EngineConstants.h"
#include "../utils/Random.h"
#include "godzillaManager.h"

static float const UFO_SUCK_MAX = float(1.4142135623730950488016887242097*CUfoManager::UFO_SUCK_SIDE_SIZE_HALF);
static D3DXVECTOR3 const LOWER_GODZILLA = D3DXVECTOR3( 0,0,-CGodzilla::HEIGHT );
int CUfo::SUCK_COUNT = 0;
int CUfo::HIT_DAMAGE = 35;

/*
template<typename _T>
struct MOO
{
	template<typename _A>
	static bool boo( _A& a );
};

MOO<int>::boo<float>( 1 );
*/

CUfo::CUfo( D3DXVECTOR3 const& pos, CMesh& mesh )
:	mPosition( pos ),
	mMesh(&mesh),
	mSucked(0),
	mEnergy( CUfo::INITIAL_ENERGY ),
	mState( S_RESTING ),
	mNeedRelease( false ),
	mScale( 1.0f ),
	mGodzillaMade( false ),
	mGodzilla( NULL )
{
}

void CUfo::update()
{
	D3DXMATRIX tm, sm, rm;

	D3DXMatrixRotationX( &rm, D3DX_PI/2.0f );
	D3DXMatrixTranslation( &tm, mPosition.x, mPosition.y, mPosition.z );
	D3DXMatrixScaling( &sm, mScale, mScale, mScale );

	mMesh->getWorldMatrix() = rm*sm*tm;
}

void CUfo::dropGodzilla()
{
	assert( mGodzilla );

	mGodzilla->flying( false );
	mGodzilla = NULL;
}

//
//
//
template<typename _T, typename _T1, typename _T2>
inline void saturate( _T& arg, _T1 const& _min, _T2 const& _max )
{
	if( arg < _min ) arg = _min;
	if( arg > _max ) arg = _max;
}

static float speedCorrection( float dist, float speed )
{
	float const _min = 25;
	float k = powf( dist, 1.2f ) / 5000;
	return _min + speed * k;
}

inline bool overTerrain( D3DXVECTOR3 const& pos )
{
	if( pos.x > 0 && pos.x <= gcon::WORLD_X &&
		pos.y > 0 && pos.y <= gcon::WORLD_Y )
		return true;
	else
		return false;
}

bool inRadius( D3DXVECTOR3 const& p1, D3DXVECTOR3 const& p0, float dist )
{
	float dist_to_point = D3DXVec3Length( &(p1-p0) );
	return dist_to_point <= dist;
}

D3DXVECTOR3 CUfo::genTerrainPoint()
{
	D3DXVECTOR3 rnd = random::randv3s( FROM_TERRAIN_CENTER_MAX );
	rnd.z = 0;
	return
		D3DXVECTOR3( gcon::WORLD_X/2, gcon::WORLD_X/2, MIN_Z + random::randf(100) ) +
		rnd;
}

D3DXVECTOR3 CUfo::genAwayPoint()
{
	int rnd = rand();
	return
	D3DXVECTOR3(
		(rnd&0x01 ? MIN_AWAY_DIST : -MIN_AWAY_DIST) + (rnd&0x02 ? random::randf(RAND_AWAY_DIST) : -random::randf(RAND_AWAY_DIST)),
		(rnd&0x04 ? MIN_AWAY_DIST : -MIN_AWAY_DIST) + (rnd&0x08 ? random::randf(RAND_AWAY_DIST) : -random::randf(RAND_AWAY_DIST)),
		MIN_Z + random::randf(MIN_Z) );
}

void CUfo::movement( CFrameTime const& ft )
{
	switch( mState )
	{
	case S_RESTING:
		{
			mSuckPoint = genTerrainPoint();
			mEnergy = INITIAL_ENERGY;
			mSucked = 0;
			mNeedTransform = 0;

			mState = S_FLYING_TO_SUCK;
			mGodzillaMade = false;
			mNeedRelease = false;
			break;
		}
	case S_FLYING_TO_SUCK:
		{
			flyTo( ft, mSuckPoint );
			if( inRadius( getPosition(), mSuckPoint, 10.0f ) ) {
				mNeedRelease = false;
				mState = S_SUCKING;
			}
			break;
		}
	case S_SUCKING:
		{
			if( mSucked >= SUCK_COUNT  )
			{
				mState = S_GOING_HOME;
				mNeedRelease = false;
				mHomePoint = CUfo::genAwayPoint();
			}
			break;
		}
	case S_GOING_HOME:
		{
			flyTo( ft, mHomePoint );

			if( inRadius( getPosition(), mHomePoint, 10.0f ) )
			{
				if( mSucked >= SUCK_COUNT )
				{
					if( mGodzillaMade )
					{
						mGodzillaMade = false;
						mState = S_FLYING_TO_SUCK;
					}
					else
					{
						mState = S_MAKING_GODZILLA;
						mGodzillaMakingStart = ft.getTime();
					}
				}
				else
				{
					// ufo was hit and nothing was brought
					mState = S_FLYING_TO_SUCK;
					mSuckPoint = genTerrainPoint();
				}
			}
			break;
		}
	case S_MAKING_GODZILLA:
		{
			float elapsed = ft.getTime() - mGodzillaMakingStart;

			if( elapsed >= GODZILLA_MAKING_TIME )
			{
				// every entity has to be transformed
				if( mNeedTransform <= 0 )
					break;

				CGodzilla* godz = new CGodzilla;
				assert( godz );

				godz->speed( 30 );
				godz->flying( true );
				godz->scale( 2 );
				CMyGame::getInstance().mGodzillaManager.add( *godz );
				attachGodzilla( *godz );

				mGodzillaMade = true;
				mState = S_BRINGING_GODZILLA;
				mGodzillaPoint = genTerrainPoint();

				mNeedTransform = mSucked;
				mSucked = 0;
			}

			if( mNeedTransform <= 0 )
			{
				mNeedTransform = mSucked;
				/*
				float k = SUCK_COUNT / GODZILLA_MAKING_TIME;
				mNeedTransform = SUCK_COUNT * ft.getDelta();

				if( mNeedTransform > mSucked )
					mNeedTransform = mSucked;

				mSucked -= mNeedTransform;

				if( mSucked <= 10 ) {
					mNeedTransform = mSucked;
					mSucked = 0;
				}*/
			}

			break;
		}
	case S_BRINGING_GODZILLA:
		{
			flyTo( ft, mGodzillaPoint );
			if( inRadius( getPosition(), mGodzillaPoint, 10.0f ) )
			{
				// drop godzilla!
				dropGodzilla();

				mHomePoint = genAwayPoint();
				mState = S_GOING_HOME;

				// re-init
				mSucked = 0;
				mNeedRelease = false;
			}
			break;
		}
	case S_DYING:
		{			
			dropTo( ft, mGravePoint );
			if( inRadius( getPosition(), mGravePoint, 10.0f ) )
			{
				if( ft.getTime() - mGraveStartTime >= GRAVE_TIME )
				{
					// LIVE AGAIN
					// mHits.push_back( SHitInfo(ft.getTime(), ufo.getPosition(), CResourceId("ufo_hit.png") ) );
					mState = S_DEAD;
					mPosition = genAwayPoint();
				}
			}
			break;
		}
	case S_DEAD:
		{
			break;
		}
	default:
		{
			assert( !"no!" );
		}
		break;
	}
}

void CUfo::flyTo( CFrameTime const& ft, D3DXVECTOR3 const& dest )
{
	D3DXVECTOR3 pos = mPosition;
	D3DXVECTOR3 dir = dest - mPosition;
	D3DXVec3Normalize( &dir, &dir );
	
	float dist_to_point = D3DXVec3Length( &(dest-pos) );
	float speed = speedCorrection( dist_to_point, mSpeed );
	pos += dir * speed * ft.getDelta();

	if( pos.z < MIN_Z ) pos.z = MIN_Z;

	mPosition = pos;
}

void CUfo::dropTo( CFrameTime const& ft, D3DXVECTOR3 const& dest )
{
	D3DXVECTOR3 pos = mPosition;
	D3DXVECTOR3 dir = dest - mPosition;
	D3DXVec3Normalize( &dir, &dir );

	pos += dir*5;
	
	if( pos.z < dest.z ) pos.z = dest.z;

	mPosition = pos;
}

bool CUfo::wasHit( D3DXVECTOR3 const& origin, D3DXVECTOR3 const& dir )
{
	return D3DXSphereBoundProbe(
		&mPosition,
		CUfo::UFO_RADIUS,
		&origin,
		&dir ) != 0;
}

//
//
//
void CUfoManager::putUfoSuck( CMyGame::TGrid& grid, CUfo& ufo, int cx, int cy )
{
	int x0 = cx - UFO_SUCK_SIDE_SIZE_HALF;
	int y0 = cy - UFO_SUCK_SIDE_SIZE_HALF;
	int x1 = cx + UFO_SUCK_SIDE_SIZE_HALF;
	int y1 = cy + UFO_SUCK_SIDE_SIZE_HALF;

	if( x0 < 0 ) x0=0;
	if( x0 >= grid.getSecsX() ) x0 = grid.getSecsX()-1;

	if( x1 < 0 ) x1=0;
	if( x1 >= grid.getSecsX() ) x1 = grid.getSecsX()-1;

	if( y0 < 0 ) y0=0;
	if( y0 >= grid.getSecsY() ) y0 = grid.getSecsY()-1;

	if( y1 < 0 ) y1=0;
	if( y1 >= grid.getSecsY() ) y1 = grid.getSecsY()-1;

	static int n = 5;
	
	if( n-- <= 0 )
	{
		for( int y = y0; y<=y1; ++y )
		{
			for( int x = x0; x<=x1; ++x )
			{
				//float k = UFO_SUCK_MAX - sqrtf( (x-cx)*(x-cx) + (y-cy)*(y-cy) ) / UFO_SUCK_MAX;

				CMyGame::TGrid::TSector& sec = grid.getSectorByIndex( x, y );
				sec.mUfo = &ufo;
				sec.mUfoFactor += 1;//k*400;
			}
		}

		CMyGame::TGrid::TSector& sec = grid.getSectorByIndex( cx, cy );
		sec.mUfo = &ufo;
		sec.mUfoFactor = 2;//k*400;
	}
}

void CUfoManager::update( CFrameTime const& ft )
{
	// clear ufo from every sector
	for( int i = 0; i<CMyGame::getInstance().mGrid.getSecsCount(); ++i )
	{
		SUfoSector& sec = CMyGame::getInstance().mGrid.getSectorByIndex( i );

		sec.mUfo = NULL;
		sec.mUfoFactor = 0;
	}

	// put new info
	for( TUfoContainer::iterator it=mUfos.begin(); it<mUfos.end(); ++it )
	{
		if( (*it)->state() == CUfo::S_DEAD ) {
			 mHits.push_back( SHitInfo(ft.getTime(), (*it)->getPosition(), CResourceId("respawn.png") ) );
			(*it)->state( CUfo::S_RESTING );
			continue;
		}

		// update ufo positions
		(*it)->movement( ft );

		if( (*it)->godzilla() )
			(*it)->godzilla()->position( (*it)->getPosition() + LOWER_GODZILLA );

		// update grid sectors info
		int ix, iy;
		D3DXVECTOR3& pos = (*it)->getPosition();
		CMyGame::TGrid& grid = CMyGame::getInstance().mGrid;

		if( pos.x >= 0 && pos.x < gcon::WORLD_X &&
			pos.y >= 0 && pos.y < gcon::WORLD_Y )
		{
			grid.position2Index( pos.x, pos.y, ix, iy );

			if( (*it)->state() == CUfo::S_SUCKING )
				putUfoSuck( grid, **it, ix, iy );
		}
		else
		{
			// fly away
		}

		(*it)->update();
	}
}

void CUfoManager::shoot( CFrameTime const& ft, D3DXVECTOR3 const& origin, D3DXVECTOR3 const& dir )
{
	for( TUfoContainer::iterator it = mUfos.begin(); it != mUfos.end(); ++it )
	{
		CUfo& ufo = **it;

		if( ufo.wasHit( origin, dir	) )
		{
			hitUfo( ft, ufo );
			return;
		}
	}
}

void CUfoManager::hitUfo( CFrameTime const& ft, CUfo& ufo )
{
	// can damage ufo, only when sucking
	if( ufo.state() == CUfo::S_SUCKING ||
		(ufo.state() == CUfo::S_GOING_HOME && overTerrain(ufo.getPosition()) ))
	{
		// TODO: show some boom sprite

		ufo.sucked( 0 );

		ufo.damage( CUfo::HIT_DAMAGE );
		int energy = ufo.getEnergy();

		if( energy <= 0 )
		{
			// dead
			mHits.push_back( SHitInfo( ft.getTime(), ufo.getPosition(), CResourceId("dead_hit.png") ) );

			// release guys, destroy UFO
			ufo.needRelease( true );
			ufo.state( CUfo::S_DYING );
			ufo.mGravePoint = CUfo::genTerrainPoint();
			ufo.mGravePoint.z = CMyGame::getInstance().getTerrain().getAltitude( ufo.getPosition().x, ufo.getPosition().y );
			ufo.mGraveStartTime = ft.getTime();
		}
		else
		{
			// hit
			mHits.push_back( SHitInfo(ft.getTime(), ufo.getPosition(), CResourceId("ufo_hit.png") ) );

			// release guys, and fly to suck at anoter place
			ufo.needRelease( true );
			ufo.mSuckPoint = CUfo::genTerrainPoint();
			ufo.state( CUfo::S_FLYING_TO_SUCK );
		}

		ufo.sucked( 0 );
	}
	else
	{
		// TODO: show some shield sprite
		mHits.push_back( SHitInfo(ft.getTime(), ufo.getPosition(), CResourceId("no_hit.png") ) );
	}
}
