// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __COLLIDER_ODE_H
#define __COLLIDER_ODE_H

#include <ode/ode.h>
#include <ode/dTriList.h>
#include "../math/Vector3.h"
#include "../math/Plane.h"
#include "CollisionMesh.h"
#include "CollisionListener.h"
#include "../utils/AbstractNotifier.h"
//#include "../physics-ode/Physicsable.h"


namespace dingus {



// --------------------------------------------------------------------------

class CCollidable : public boost::noncopyable {
public:
	typedef dGeomID		TCollidableID;
	
	enum eClass {
		COL_SPHERE = dSphereClass,
		COL_BOX = dBoxClass,
		COL_CAPSULE = dCCylinderClass,
		COL_CYLINDER = dCylinderClass,
		COL_PLANE = dPlaneClass,
		COL_TRANSFORM = dGeomTransformClass,
		COL_RAY = dRayClass,
		COL_MESH = dTriMeshClass,
		COL_SIMPLE_CONT = dSimpleSpaceClass,
		COL_HASH_CONT = dHashSpaceClass
	};

public:
	virtual ~CCollidable() { if( mID ) dGeomDestroy(mID); }
	
	TCollidableID	getID() const { return mID; }
	operator TCollidableID() const { return mID; }
	bool operator<( const CCollidable& c ) const { return (unsigned int)mID < (unsigned int)c.mID; }
	
	void	setUserData( void *data ) { mUserData = data; }
	void*	getUserData() const { return mUserData; }

	void	destroy() { if(mID) dGeomDestroy(mID); mID = 0; }
	
	eClass	getClass() const { return (eClass)dGeomGetClass(mID); }
	
	//void	setBody( CPhysicsable::TBodyID b ) { dGeomSetBody(mID,b); }
	//CPhysicsable::TBodyID getBody() const { return dGeomGetBody(mID); }

	void	setPosition( const SVector3& pos ) { dGeomSetPosition(mID,pos.x,pos.y,pos.z); }
	SVector3 getPosition() const { const dReal* v = dGeomGetPosition(mID); return SVector3((float)v[0],(float)v[1],(float)v[2]); }

	void	setRotation( const SMatrix4x4& rot );
	void	getRotation( SMatrix4x4& dest ) const;

	void	getAABB( SVector3& vmin, SVector3& vmax ) const {
		double aabb[6];
		dGeomGetAABB( mID, aabb );
		vmin.set( (float)aabb[0], (float)aabb[2], (float)aabb[4] );
		vmin.set( (float)aabb[1], (float)aabb[3], (float)aabb[5] );
	}
	
	bool	isSpace() const { return dGeomIsSpace(mID) ? true : false; }
	
	void	setCategoryBits( unsigned long bits ) { dGeomSetCategoryBits(mID, bits); }
	void	setCollideBits( unsigned long bits ) { dGeomSetCollideBits(mID, bits); }
	unsigned long getCategoryBits() const { return dGeomGetCategoryBits(mID); }
	unsigned long getCollideBits() const { return dGeomGetCollideBits(mID); }
	
	void	enable() { dGeomEnable(mID); }
	void	disable() { dGeomDisable(mID); }
	int		isEnabled() const { return dGeomIsEnabled(mID); }
	
	void addListener( ICollisionListener& listener ) { mListeners.addListener( listener ); }
	void removeListener( ICollisionListener& listener ) { mListeners.removeListener( listener ); }
	void notifyListeners( CCollidable& him ) { mListeners.notify( *this, him ); }

	/** @return number of contacts. */
	int collideWith( CCollidable& other, int maxContacts, dContactGeom* contacts ) {
		return dCollide( mID, other, maxContacts, contacts, sizeof(dContactGeom) );
	}

private:
	class CCollisionNotifier : public dingus::CSimpleNotifier<ICollisionListener> {
	public:
		void notify( CCollidable& me, CCollidable& him ) {
			TListenerVector::iterator it, itEnd = getListeners().end();
			for( it = getListeners().begin(); it != itEnd; ++it ) {
				assert( *it );
				(*it)->onCollide( me, him );
			}
		}
	};

protected:
	CCollidable() : mID(0), mUserData(NULL) { }

	TCollidableID		mID;
	CCollisionNotifier	mListeners;
	void*				mUserData;
};



// --------------------------------------------------------------------------

class CCollidableContainer : public CCollidable {
public:
	typedef dSpaceID	TContainerID;

public:
	TContainerID getID() const { return (TContainerID)mID; }
	operator TContainerID() const { return (TContainerID) mID; }
	
	void setCleanup( bool cleanup ) { dSpaceSetCleanup( getID(), cleanup?1:0 ); }
	bool isCleanup() const { return dSpaceGetCleanup(getID()) ? true:false; }
	
	void add( CCollidable& x ) { dSpaceAdd(getID(),x); }
	void remove( CCollidable& x ) { dSpaceRemove(getID(),x); }
	bool contains( CCollidable const& x ) const { return dSpaceQuery(getID(),x) ? true:false; }
	
	int size() const { return dSpaceGetNumGeoms(getID()); }
	TCollidableID getCollidable( int i ) { return dSpaceGetGeom(getID(),i); }
	
	void collideSelf( void *data, dNearCallback *callback ) { dSpaceCollide(getID(),data,callback); }

protected:
	// instance subclasses
	CCollidableContainer() { }
};



// --------------------------------------------------------------------------

class CSimpleCollidableContainer : public CCollidableContainer {
public:
	CSimpleCollidableContainer( CCollidableContainer::TContainerID parent ) {
		mID = (TCollidableID)dSimpleSpaceCreate(parent);
		dGeomSetData(mID, this);
	}
};



// --------------------------------------------------------------------------

class CHashCollidableContainer : public CCollidableContainer {
public:
	CHashCollidableContainer( CCollidableContainer::TContainerID parent ) {
		mID = (TCollidableID)dHashSpaceCreate(parent);
		dGeomSetData(mID, this);
	}
	
	void setLevels( int minlevel, int maxlevel ) { dHashSpaceSetLevels(getID(),minlevel,maxlevel); }
};



// --------------------------------------------------------------------------

class CCollidableSphere : public CCollidable {
public:
	//CCollidableSphere() { }
	CCollidableSphere( CCollidableContainer::TContainerID parent, float radius ) {
		mID = dCreateSphere(parent,radius);
		dGeomSetData(mID, this);
	}
	
	/*
	void create( CCollidableContainer::TContainerID parent, float radius ) {
		if(mID) dGeomDestroy(mID);
		mID = dCreateSphere(parent,radius);
		dGeomSetData(mID, this);
	}
	*/
	
	void setRadius( float radius ) { dGeomSphereSetRadius(mID, radius); }
	float getRadius() const { return (float)dGeomSphereGetRadius(mID); }
};



// --------------------------------------------------------------------------

class CCollidableBox : public CCollidable {
public:
	//CCollidableBox() { }
	CCollidableBox( CCollidableContainer::TContainerID parent, const SVector3& len ) {
		mID = dCreateBox(parent,len.x,len.y,len.z);
		dGeomSetData(mID, this);
	}

	/*	
	void create( CCollidableContainer::TContainerID parent, const SVector3& len ) {
		if(mID) dGeomDestroy (mID);
		mID = dCreateBox(parent,len.x,len.y,len.z);
		dGeomSetData(mID, this);
	}
	*/
	
	void setLen( const SVector3& len ) { dGeomBoxSetLengths(mID,len.x,len.y,len.z); }
	SVector3 getLen() const { dVector3 l; dGeomBoxGetLengths(mID,l); return SVector3((float)l[0],(float)l[1],(float)l[2]); }
};



// --------------------------------------------------------------------------

class CCollidablePlane : public CCollidable {
public:
	//CCollidablePlane() { }
	CCollidablePlane( CCollidableContainer::TContainerID parent, const SPlane& p ) {
		mID = dCreatePlane(parent,p.a,p.b,p.c,p.d);
		dGeomSetData(mID,this);
	}

	/*
	void create (CCollidableContainer& cont, float a, float b, float c, float d) {
		if (mID) dGeomDestroy (mID);
		mID = dCreatePlane (cont,a,b,c,d);
		dGeomSetData(mID, this);
	}*/
	
	void setPlane( const SPlane& p ) { dGeomPlaneSetParams(mID, p.a, p.b, p.c, p.d); }
	void getParams( SPlane& res ) const { dVector3 r; dGeomPlaneGetParams(mID,r); res.a=(float)r[0];res.b=(float)r[1];res.c=(float)r[2];res.d=(float)r[3]; }
};



// --------------------------------------------------------------------------
/*
class CCollidableCapsule : public CCollidable {
public:
	//CCollidableCapsule() { }
	CCollidableCapsule( CCollidableContainer::TContainerID parent, float radius, float length ) {
		mID = dCreateCCylinder (parent,radius,length);
		dGeomSetData(mID, this);
	}
	
	void create( CCollidableContainer::TContainerID parent, float radius, float length ) {
		if(mID) dGeomDestroy (mID);
		mID = dCreateCCylinder(parent,radius,length);
		dGeomSetData(mID, this);
	}
	
	void setParams( float radius, float length ) { dGeomCCylinderSetParams(mID,radius,length); }
	void getParams( float& radius, float& length) const { dGeomCCylinderGetParams(mID,&radius,&length); }
};
*/


// --------------------------------------------------------------------------

class CCollidableRay : public CCollidable {
public:
	//CCollidableRay() { }
	CCollidableRay( CCollidableContainer::TContainerID parent, float length ) {
		mID = dCreateRay(parent,length);
		dGeomSetData(mID, this);
	}

	/*
	void create( CCollidableContainer::TContainerID parent, float length ) {
		if(mID) dGeomDestroy (mID);
		mID = dCreateRay(parent,length);
	}
	*/

	void	setLength( float length ) { dGeomRaySetLength(mID,length); }
	float	getLength() const { return (float)dGeomRayGetLength(mID); }
	
	void	setParams( const SVector3& pos, const SVector3& dir ) { dGeomRaySet(mID,pos.x,pos.y,pos.z,dir.x,dir.y,dir.z); }
	void	getParams( SVector3& pos, SVector3& dir ) const;
};



// --------------------------------------------------------------------------

class CCollidableMesh : public CCollidable {
public:
	//CCollidableMesh() : mMesh(0) { }
	CCollidableMesh( CCollidableContainer::TContainerID parent, const CCollisionMesh& mesh, bool flipCull ) {
		initMesh( parent, mesh, flipCull );
		dGeomSetData(mID, this);
	}

	/*
	void create( CCollidableContainer::TContainerID parent, const CCollisionMesh& mesh ) {
		if(mID) dGeomDestroy (mID);
		initMesh(parent,mesh);
	}
	*/

	void setFlipCull( bool flipCull ) { dGeomTriListSetFlipCull(mID,flipCull); }

private:
	void	initMesh( CCollidableContainer::TContainerID parent, const CCollisionMesh& mesh, bool flipCull );
private:
	CCollisionMesh const*	mMesh;
};



// --------------------------------------------------------------------------

/*
class CCollidableTransform : public CCollidable {
public:
	//CCollidableTransform() { }
	CCollidableTransform( CCollidableContainer::TContainerID parent ) { mID = dCreateGeomTransform(parent); }
	
	void create( CCollidableContainer::TContainerID parent ) {
		if(mID) dGeomDestroy(mID);
		mID = dCreateGeomTransform(parent);
	}
	
	void setCollidable( TCollidableID coll ) { dGeomTransformSetGeom(mID, coll); }
	TCollidableID getCollidable() const { return dGeomTransformGetGeom(mID); }
	
	void setCleanup( bool cleanup )	{ dGeomTransformSetCleanup(mID,cleanup?1:0); }
	bool isCleanup() const { return dGeomTransformGetCleanup(mID)?true:false; }
	
	//void setInfo (int mode) { dGeomTransformSetInfo (mID,mode); }
	//int getInfo() const { return dGeomTransformGetInfo (mID); }
};
*/
	
	
}; // namespace


#endif
