// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "InertiaTensorBuilder.h"

using namespace dingus;


void CInertiaTensorBuilder::compute( CCollisionMesh& mesh )
{
	const SVector3* vertices = mesh.getVertices();
	const int* indices = mesh.getIndices();
	const int triCount = mesh.getNumIndices()/3;

	const double mult[10] = { 1/6.0, 1/24.0, 1/24.0, 1/24.0, 1/60.0, 1/60.0, 1/60.0, 1/120.0, 1/120.0, 1/120.0 };
	double intg[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }; // order: 1, x, y, z, x^2, y^2, z^2, xy, yz, zx

	for( int t = 0; t < triCount; ++t ) {
		// get vertices of triangle t
		int i0 = indices[3*t  ]; 
		int i1 = indices[3*t+1]; 
		int i2 = indices[3*t+2];

		float x0 = vertices[i0].x; 
		float y0 = vertices[i0].y; 
		float z0 = vertices[i0].z;

		float x1 = vertices[i1].x; 
		float y1 = vertices[i1].y; 
		float z1 = vertices[i1].z;

		float x2 = vertices[i2].x; 
		float y2 = vertices[i2].y; 
		float z2 = vertices[i2].z;

		// get edges and cross product of edges
		float a1 = x1-x0; 
		float b1 = y1-y0; 
		float c1 = z1-z0; 
		
		float a2 = x2-x0; 
		float b2 = y2-y0; 
		float c2 = z2-z0;

		float d0 = b1 * c2 - b2 * c1; 
		float d1 = a2 * c1 - a1 * c2; 
		float d2 = a1 * b2 - a2 * b1;

		float f1x, f1y, f1z;
		float f2x, f2y, f2z;
		float f3x, f3y, f3z;

		float g0x, g0y, g0z;
		float g1x, g1y, g1z;
		float g2x, g2y, g2z;
		
		// compute integral terms
		subExpressions( x0, x1, x2, f1x, f2x, f3x, g0x, g1x, g2x );
		subExpressions( y0, y1, y2, f1y, f2y, f3y, g0y, g1y, g2y );
		subExpressions( z0, z1, z2, f1z, f2z, f3z, g0z, g1z, g2z );

		// update integrals
		intg[0] += d0 * f1x;
		intg[1] += d0 * f2x; 
		intg[2] += d1 * f2y; 
		intg[3] += d2 * f2z;
		intg[4] += d0 * f3x; 
		intg[5] += d1 * f3y; 
		intg[6] += d2 * f3z;
		intg[7] += d0 * (y0 * g0x + y1 * g1x + y2 * g2x);
		intg[8] += d1 * (z0 * g0y + z1 * g1y + z2 * g2y);
		intg[9] += d2 * (x0 * g0z + x1 * g1z + x2 * g2z);
	}

	for( int i = 0; i < 10; ++i )
		intg[i] *= mult[i];

	// volume
	double volume = intg[0];

	// center of mass
	SVector3 massCenter( (float)intg[1], (float)intg[2], (float)intg[3] );
	massCenter /= (float)volume;

	// inertia tensor relative to center of mass {xx, yy, zz, xy, yz, xz}
	SMatrix4x4 inTensor;
	inTensor.identify();
	inTensor._11 = (float)(intg[5] + intg[6] - volume * ( massCenter.y * massCenter.y + massCenter.z * massCenter.z ));
	inTensor._22 = (float)(intg[4] + intg[6] - volume * ( massCenter.z * massCenter.z + massCenter.x * massCenter.x ));
	inTensor._33 = (float)(intg[4] + intg[5] - volume * ( massCenter.x * massCenter.x + massCenter.y * massCenter.y ));
	inTensor._12 = inTensor._21 = (float)(-( intg[7] - volume * massCenter.x * massCenter.y ));
	inTensor._23 = inTensor._32 = (float)(-( intg[8] - volume * massCenter.y * massCenter.z ));
	inTensor._13 = inTensor._31 = (float)(-( intg[9] - volume * massCenter.z * massCenter.x ));

	mesh.setMassProps( inTensor, massCenter, (float)volume );
}

void CInertiaTensorBuilder::subExpressions( float w0, float w1, float w2, float& f1, float& f2, float& f3, float& g0, float& g1, float& g2 )
{
	float temp0 = w0 + w1; 
	float temp1 = w0 * w0; 
	float temp2 = temp1 + w1 * temp0;

	f1 = temp0 + w2; 
	f2 = temp2 + w2 * f1; 
	f3 = temp1 * w0 + w1 * temp2 + w2 * f2;

	g0 = f2 + w0 * ( f1 + w0 ); 
	g1 = f2 + w1 * ( f1 + w1 ); 
	g2 = f2 + w2 * ( f1 + w2 );
}
