// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef D3D_APPLICATION_H__
#define D3D_APPLICATION_H__

#include "D3DEnumeration.h"
#include "D3DSettings.h"


namespace dingus {
	
	

//---------------------------------------------------------------------------
// error codes

enum eAppMsg { NONE = 0, APPMUSTEXIT, SWITCHEDTOREF };

enum eAppErr {
	NODIRECT3D			= 0x82000001,
	NOWINDOW			= 0x82000002,
	NOCOMPATIBLEDEVICES = 0x82000003,
	NOWINDOWABLEDEVICES = 0x82000004,
	NOHARDWAREDEVICE	= 0x82000005,
	HALNOTCOMPATIBLE	= 0x82000006,
	NOWINDOWEDHAL		= 0x82000007,
	NODESKTOPHAL		= 0x82000008,
	NOHALTHISMODE		= 0x82000009,
	NONZEROREFCOUNT 	= 0x8200000a,
	MEDIANOTFOUND		= 0x8200000b,
	RESETFAILED 		= 0x8200000c,
	NULLREFDEVICE		= 0x8200000d
};



//---------------------------------------------------------------------------

/**
*  CD3DApplication - framework for D3D apps.
*
*  Override the following functions:
*		initialize()		- one time initialization.
*		initDeviceObjects()
*		frameMove()
*		render()
*		deleteDeviceObjects()
*		shutdown()
*		msgProc()
*/
class CD3DApplication {
protected:
	CD3DEnumeration   mEnumeration;
	CD3DSettings	  mSettings;
	
	// internal variables for the state of the app
	bool			  mWindowed;
	bool			  mActive;
	bool			  mDeviceLost;
	bool			  mMinimized;
	bool			  mMaximized;
	bool			  mIgnoreSizeChange;
	bool			  mDeviceObjectsInited;
	bool			  mDeviceObjectsRestored;
	
	// internal variables used for timing
	bool			  mFrameMoving;
	bool			  mSingleStep;
	
protected:
	// internal error handling function
	HRESULT displayErrorMsg( HRESULT hr, eAppMsg type );
	
	// internal functions to manage and render the 3D scene
	static bool checkDeviceHelper( const D3DCAPS9& caps, eVertexProcessing vertexProcessing, D3DFORMAT backBufferFormat );
	void	buildPresentParamsFromSettings();
	bool	findBestWindowedMode( bool requireHAL, bool requireREF );
	bool	findBestFullscreenMode( bool requireHAL, bool requireREF );
	HRESULT chooseInitialD3DSettings();
	HRESULT initialize3DEnvironment();
	HRESULT handlePossibleSizeChange();
	HRESULT reset3DEnvironment();
	HRESULT toggleFullscreen();
	HRESULT forceWindowed();
	HRESULT userSelectNewDevice();
	void	cleanup3DEnvironment();
	HRESULT render3DEnvironment();
	virtual HRESULT adjustWindowForChange();
	virtual void updateStats();
	
protected:
	// main objects used for creating and rendering the 3D scene
	D3DPRESENT_PARAMETERS mPresentParams;  // Parameters for CreateDevice/Reset
	HWND				mHWnd;				// The main app window
	HWND				mHWndFocus; 		// The D3D focus window (usually same as m_hWnd)
	HMENU				mHMenu; 			// App menu bar (stored here when fullscreen)
	IDirect3D9* 		mD3D;			   // The main D3D object
	IDirect3DDevice9*	mD3DDevice; 	   // The D3D rendering device
	D3DCAPS9			mD3DCaps;			// Caps for the device
	D3DSURFACE_DESC 	mBackBuffer;   // Surface desc of the backbuffer
	DWORD				mCreateFlags;	  // Indicate sw or hw vertex processing
	DWORD				mWindowStyle;	  // Saved window style for mode switches
	RECT				mWindowBounds;	  // Saved window bounds for mode switches
	RECT				mWindowClient;	  // Saved client area size for mode switches
	
	// variables for timing
	double	mTime;			 // Current time in seconds
	double	mElapsedTime;	   // Time elapsed since last frame
	float	mFPS;			   // Instanteous frame rate
	TCHAR	mDeviceStats[90];// String to hold D3D device stats
	TCHAR	mFrameStats[90]; // String to hold frame stats
	
	// overridable variables for the app
	const TCHAR*	mWindowTitle;    // Title for the app's window
	DWORD			mCreationWidth;   // Width used to create window
	DWORD			mCreationHeight;  // Height used to create window
	bool			mShowCursorWhenFullscreen; // Whether to show cursor when fullscreen
	bool			mClipCursorWhenFullscreen; // Whether to limit cursor pos when fullscreen
	bool			mStartFullscreen;  // Whether to start up the app in fullscreen mode
	
	// overridable functions for the 3D scene created by the app
	virtual HRESULT checkDevice( const D3DCAPS9& caps, DWORD behavior, D3DFORMAT format )   { return S_OK; }
	virtual HRESULT initialize()						   { return S_OK; }
	virtual HRESULT createDeviceObjects() 					   { return S_OK; }
	virtual HRESULT activateDeviceObjects()					   { return S_OK; }
	//virtual HRESULT frameMove() 							   { return S_OK; }
	virtual HRESULT passivateDeviceObjects()				   { return S_OK; }
	virtual HRESULT deleteDeviceObjects()					   { return S_OK; }
	virtual HRESULT shutdown()							   { return S_OK; }
	
public:
	virtual HRESULT performOneTime() { return S_OK; }

	// functions to create, run, pause, and clean up the application
	virtual HRESULT create( HINSTANCE hInstance );
	virtual int 	run();
	virtual LRESULT msgProc( HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam );
	virtual void	pause( bool bPause );
	bool			isActive() const { return mActive; }
	void			close();
	virtual 		~CD3DApplication() { }
	
	CD3DApplication();
};
	
}; // namespace dingus



#endif
