// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "RenderableBillboards.h"
#include "geometry/DynamicVBManager.h"

#include "Vertices.h"
#include "../renderer/EffectSlot.h"
#include "../kernel/D3DDevice.h"

using namespace dingus;


const int SLOT_POOLLET_SIZE = 16;


CRenderableBillboards::CRenderableBillboards(
	CD3DIndexBuffer& ib, CDynamicVBManager& vbManager,
	CEffectParams::TParamName texParamName )
:	mIB( &ib ), mVBSource( vbManager, sizeof(TVertex) ),
	mTexParamName( texParamName )
{
	// add self as a listener
	addListener( *this );
}

CRenderableBillboards::~CRenderableBillboards()
{
	removeListener( *this );
}



CRenderableBillboards::TSlot& CRenderableBillboards::getSlot( CD3DTexture& tex )
{
	TTextureSlotMap::iterator it = mSlots.find( &tex );
	if( it == mSlots.end() ) {
		it = mSlots.insert( std::make_pair( &tex, TSlot(SLOT_POOLLET_SIZE) ) ).first;
	}
	return it->second;
}


SBillboard& CRenderableBillboards::addBill( CD3DTexture& texture )
{
	TSlot& slot = getSlot( texture );
	SBillboard& bill = slot.add();
	bill.texture = &texture;
	bill.discarded = false;
	return bill;
}


// ------------------------------------------------------------------
//  Rendering


void CRenderableBillboards::beforeRender( CRenderable& r, CEffectSlot const& fx )
{
	assert( mChunks.empty() );
	
	// set IB
	assert( mIB );
	gD3DDevice->SetIndices( mIB->getObject() );
	//ib->getObject()->Release();

	// render into VB
	TTextureSlotMap::iterator it, itEnd = mSlots.end();
	for( it = mSlots.begin(); it != itEnd; ++it ) {
		CD3DTexture* tex = it->first;
		assert( tex );
		TSlot& slot = it->second;
		if( slot.empty() )
			continue;

		CVBChunk::TSharedPtr chunk = mVBSource.lock( slot.size() * 4 ); // 4 verts per billboard
		TVertex* vb = reinterpret_cast<TVertex*>( chunk->getData() );
		TSlot::iterator bit, bitEnd = slot.end();
		int ttt= 0;
		for( bit = slot.begin(); bit != bitEnd; /**/ ) {
			const SBillboard& bill = *bit;
			// if discarded - remove
			if( bill.discarded ) {
				bit = slot.erase(bit);
				bitEnd = slot.end();
				continue;
			}
			vb->p.x		= bill.x1; vb->p.y = bill.y1; vb->p.z = 0.1f; // TBD
			vb->diffuse	= bill.color;
			vb->tu		= bill.tu1; vb->tv = bill.tv1;
			++vb;
			vb->p.x		= bill.x2; vb->p.y = bill.y1; vb->p.z = 0.1f; // TBD
			vb->diffuse	= bill.color;
			vb->tu		= bill.tu2; vb->tv = bill.tv1;
			++vb;
			vb->p.x		= bill.x2; vb->p.y = bill.y2; vb->p.z = 0.1f; // TBD
			vb->diffuse	= bill.color;
			vb->tu		= bill.tu2; vb->tv = bill.tv2;
			++vb;
			vb->p.x		= bill.x1; vb->p.y = bill.y2; vb->p.z = 0.1f; // TBD
			vb->diffuse	= bill.color;
			vb->tu		= bill.tu1; vb->tv = bill.tv2;
			++vb;
			++bit;
			++ttt;
		}
		chunk->unlock( slot.size()*4 ); // we may have not filled discarded bills

		mChunks.push_back( std::make_pair( tex, chunk ) );
	}
}

void CRenderableBillboards::afterRender( CRenderable& r, CEffectSlot const& fx )
{
	mChunks.clear();
}


void CRenderableBillboards::renderContent( CRenderContext const& ctx, CEffectSlot const& fx )
{
	// render all chunks
	TTextureChunkVector::const_iterator it, itEnd = mChunks.end();
	for( it = mChunks.begin(); it != itEnd; ++it ) {
		CD3DTexture* tex = it->first;
		const CVBChunk& chunk = *it->second;

		// set texture on effect
		// TBD: better supply handle, not name - would be faster
		fx.getEffect().getObject()->SetTexture( mTexParamName, tex->getObject() );
		//tex->getObject()->Release(); // TBD: not sure if this is needed

		// set streams, VB, etc.
		CD3DVertexBuffer& vb = chunk.getVB();
		gD3DDevice->SetStreamSource( 0, vb.getObject(), 0, chunk.getStride() );
		//vb.getObject()->Release();
		gD3DDevice->SetVertexShader( NULL );
		gD3DDevice->SetFVF( FVF_XYZ_DIFFUSE_TEX1 );
		gD3DDevice->DrawIndexedPrimitive( D3DPT_TRIANGLELIST, chunk.getOffset(), 0, chunk.getSize(), 0, chunk.getSize()/2 );
		// stats
		ctx.getStats().incDrawCalls();
		ctx.getStats().incVerticesRendered( chunk.getSize() );
		ctx.getStats().incPrimsRendered( chunk.getSize()/2 );
	}
}
