// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __VB_CHUNK_HELPER_H
#define __VB_CHUNK_HELPER_H

#include "VBChunk.h"
#include "ChunkSource.h"


namespace dingus {



/**
 *  Helper class for managing more complex chunk locking scenarios.
 *
 *  Enables locking the whole needed portion in smaller chunks, and then
 *  rendering everything. Depending of situation, this can be faster
 *  than locking whole big portion.
 */
class CVBChunkHelper {
	typedef CVBChunk::TSharedPtr		TSharedChunk;
	typedef std::vector<TSharedChunk>	TChunkVector;

public:
	CVBChunkHelper( IChunkSource<CVBChunk>& source )
		: mSource(&source), mCurrentChunkLocked(false) { }

	/**
	 *  Lock some element count. Put the obtained chunk in the end of list for
	 *  later rendering. If there was a previous locked chunk, it is unlocked.
	 *
	 *  @param count Element (eg. vertex or index) count.
	 */
	byte* lock( int count );
	
	/**
	 *  Unlock the last locked chunk.
	 *  @param count TBD - what is the precise meaning?
	 */
	void unlock( int count = 0 ) {
		assert( mCurrentChunkLocked && !"Lock chunk first!" );
		assert( getChunk().get() );
		getChunk()->unlock( count );
		mCurrentChunkLocked = false;
	}

	/**
	 *  Clears locked chunks.
	 */
	virtual void clear() {
		assert( !mCurrentChunkLocked && !"Unlock chunk first!" );
		mChunks.clear();
	}

	/*
	virtual bool render( IVertexStream& stream, bool discardable = true ) {
		assert( mCurrentChunkLocked || "Unlock chunk first!" );

		for( TChunkVector::const_iterator chunk = mChunks.begin(); chunk != mChunks.end(); ++chunk )
		{
			assert( chunk->get() );
			if( !(*chunk)->isValid() )
				return false;
		
			(*chunk)->render( stream, discardable );
		}
		return true;
	}
	*/

	TSharedChunk getChunk() const { return mChunks.back(); }
	TSharedChunk getChunk() { return mChunks.back(); }

private:
	// disable copy
	CVBChunkHelper( CVBChunkHelper const& source );
	CVBChunkHelper& operator= ( CVBChunkHelper const& rhs );
	
private:
	IChunkSource<CVBChunk>*		mSource;
	TChunkVector	mChunks;
	bool			mCurrentChunkLocked;
};


}; // namespace


#endif
