// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __PARTICLE_PHYSICS_H
#define __PARTICLE_PHYSICS_H

namespace dingus {


// --------------------------------------------------------------------------

template< typename PARTICLE >
class IParticlePhysics {
public:
	virtual ~IParticlePhysics() { };
	virtual void applyTo( PARTICLE& p ) const = 0;
};


// --------------------------------------------------------------------------

template< typename PARTICLE >
class CParticleCompositePhysics : public IParticlePhysics<PARTICLE> {
public:
	typedef IParticlePhysics<PARTICLE>	TPhysics;
	typedef std::vector<TPhysics*>		TPhysicsVector;
public:
	CParticleCompositePhysics() { };
	CParticleCompositePhysics( const TPhysicsVector& physics ) : mPhysics( physics ) { };

	virtual void applyTo( PARTICLE& p ) const {
		TPhysicsVector::const_iterator it, itEnd = mPhysics.end();
		for( it = mPhysics.begin(); it != itEnd; ++it ) {
			(*it)->applyTo( p );
		}
	}

	void addPhysics( TPhysics& p ) { mPhysics.push_back( &p ); }

private:
	TPhysicsVector	mPhysics;
};


// --------------------------------------------------------------------------

template< typename PARTICLE >
class CParticleAnimPhysics : public IParticlePhysics<PARTICLE> {
public:
	typedef	PARTICLE::SState	TKey;
	typedef std::vector<TKey>	TKeyVector;

public:
	CParticleAnimPhysics()
		: mKeyCountMinusOne(-1) { };
	CParticleAnimPhysics( TKeyVector const& keys )
		: mKeys(keys), mKeyCountMinusOne(keys.size()-1) { };

	void	addKey( TKey const& k ) { mKeys.push_back(k); mKeyCountMinusOne = (float)mKeys.size()-1; }

	virtual void applyTo( PARTICLE& p ) const {
		float relAge = 1.0f - p.getTTL() * p.getInvLifetime();
		float index = relAge * mKeyCountMinusOne;
		float from = floorf(index);
		float frac = index - from;
		int i1 = (int)from;

		assert( i1>=0 && i1+1<mKeys.size() || !"bad key index" );
		
		TKey const& k1 = mKeys[i1];
		TKey const& k2 = mKeys[i1+1];

		p.setFromState( k1, k2, frac );
	};

private:
	TKeyVector	mKeys;
	float		mKeyCountMinusOne;
};


}; // namespace

#endif
