// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __PHYSICS_ODE_JOINT_H
#define __PHYSICS_ODE_JOINT_H


#include "../math/Vector3.h"
#include "PhysicsIDs.h"
#include "PhysicsContext.h"


namespace dingus {


// --------------------------------------------------------------------------

class CPhysicsJoint : public boost::noncopyable {
public:
	enum eType {
		JOINT_BALL = dJointTypeBall,
		JOINT_HINGE = dJointTypeHinge,
		JOINT_SLIDER = dJointTypeSlider,
		JOINT_CONTACT = dJointTypeContact,
		JOINT_UNIVERSAL = dJointTypeUniversal,
		JOINT_HINGE2 = dJointTypeHinge2,
		JOINT_FIXED = dJointTypeFixed,
		JOINT_AMOTOR = dJointTypeAMotor
	};

	enum eParam {
		PARAM_LO_STOP = dParamLoStop,
		PARAM_HI_STOP = dParamHiStop,
		PARAM_VEL = dParamVel,
		PARAM_FMAX = dParamFMax,
		PARAM_FUDGE = dParamFudgeFactor,
		PARAM_BOUNCE = dParamBounce,
		PARAM_CFM = dParamCFM,
		PARAM_STOP_ERP = dParamStopERP,
		PARAM_STOP_CFM = dParamStopCFM,
		PARAM_SUSPENSION_ERP = dParamSuspensionERP,
		PARAM_SUSPENSION_CFM = dParamSuspensionCFM,

		PARAM_VEL_2 = dParamVel2,
		PARAM_FMAX_2 = dParamFMax2,
		PARAM_CFM_2 = dParamCFM2,
	};

public:
	CPhysicsJoint() : mID(0) { }
	~CPhysicsJoint() { if(mID) dJointDestroy(mID); } // no virtual intentionally
	
	TPhysicsJointID getID() const { return mID; }
	operator TPhysicsJointID() const { return mID; }
	
	void attach( TPhysicsableID body1, TPhysicsableID body2 ) { dJointAttach(mID, body1, body2); }
	
	/*
	void setData (void *data)
	{ dJointSetData (mID, data); }
	void *getData (void *data) const
	{ return dJointGetData (mID); }
	*/
	
	eType getType() const { return (eType)dJointGetType(mID); }
	
	TPhysicsableID getBody( int index ) const { return dJointGetBody(mID, index); }

protected:
	TPhysicsJointID mID;
};



// --------------------------------------------------------------------------

class CPhysicsJointBall : public CPhysicsJoint {
public:
	CPhysicsJointBall() { }
	CPhysicsJointBall( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) { mID = dJointCreateBall(world, group); }
	
	void create( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) {
		if(mID) dJointDestroy(mID);
		mID = dJointCreateBall(world, group);
	}
	
	void setAnchor( const SVector3& a ) { dJointSetBallAnchor(mID, a.x, a.y, a.z); }
	SVector3 getAnchor() const { dVector3 r; dJointGetBallAnchor(mID,r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
};



// --------------------------------------------------------------------------

class CPhysicsJointHinge : public CPhysicsJoint {
public:
	CPhysicsJointHinge() { }
	CPhysicsJointHinge( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) { mID = dJointCreateHinge(world, group); }
	
	void create( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) {
		if(mID) dJointDestroy(mID);
		mID = dJointCreateHinge(world, group);
	}
	
	void setAnchor( const SVector3& a ) { dJointSetHingeAnchor(mID, a.x, a.y, a.z); }
	SVector3 getAnchor() const { dVector3 r; dJointGetHingeAnchor(mID,r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	
	void setAxis( const SVector3& a ) { dJointSetHingeAxis(mID, a.x, a.y, a.z); }
	SVector3 getAxis() const { dVector3 r; dJointGetHingeAxis(mID,r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	
	float getAngle() const { return (float)dJointGetHingeAngle(mID); }
	float getAngleRate() const { return (float)dJointGetHingeAngleRate(mID); }
	
	//
	// params

	void	setLoStopAngle( float value ) { setParam(PARAM_LO_STOP,value); }
	void	setHiStopAngle( float value ) { setParam(PARAM_HI_STOP,value); }
	void	setStopBounce( float value ) { setParam(PARAM_BOUNCE,value); }
	void	setStopCFM( float value ) { setParam(PARAM_STOP_CFM,value); }
	void	setStopERP( float value ) { setParam(PARAM_STOP_ERP,value); }
	void	setMotorVelocity( float value ) { setParam(PARAM_VEL,value); }
	void	setMotorMaxForce( float value ) { setParam(PARAM_FMAX,value); }
	void	setFudgeFactor( float value ) { setParam(PARAM_FUDGE,value); }
	void	setCFM( float value ) { setParam(PARAM_CFM,value); }
	void	setSuspensionCFM( float value ) { setParam(PARAM_SUSPENSION_CFM,value); }
	void	setSuspensionERP( float value ) { setParam(PARAM_SUSPENSION_ERP,value); }

	float	getLoStopAngle() const { return getParam(PARAM_LO_STOP); }
	float	getHiStopAngle() const { return getParam(PARAM_HI_STOP); }
	float	getStopBounce() const { return getParam(PARAM_BOUNCE); }
	float	getStopCFM() const { return getParam(PARAM_STOP_CFM); }
	float	getStopERP() const { return getParam(PARAM_STOP_ERP); }
	float	getMotorVelocity() const { return getParam(PARAM_VEL); }
	float	getMotorMaxForce() const { return getParam(PARAM_FMAX); }
	float	getFudgeFactor() const { return getParam(PARAM_FUDGE); }
	float	getCFM() const { return getParam(PARAM_CFM); }
	float	getSuspensionCFM() const { return getParam(PARAM_SUSPENSION_CFM); }
	float	getSuspensionERP() const { return getParam(PARAM_SUSPENSION_ERP); }

private:
	void setParam( eParam param, float value ) { dJointSetHingeParam(mID, param, value); }
	float getParam( eParam param ) const { return (float)dJointGetHingeParam(mID, param); }
};


// --------------------------------------------------------------------------

class CPhysicsJointSlider : public CPhysicsJoint {
public:
	CPhysicsJointSlider() { }
	CPhysicsJointSlider( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) { mID = dJointCreateSlider(world, group); }
	
	void create( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) {
		if(mID) dJointDestroy (mID);
		mID = dJointCreateSlider(world, group);
	}
	
	void setAxis( const SVector3& a ) { dJointSetSliderAxis(mID, a.x, a.y, a.z); }
	SVector3 getAxis() const { dVector3 r; dJointGetSliderAxis(mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	
	float getPosition() const { return (float)dJointGetSliderPosition(mID); }
	float getPositionRate() const { return (float)dJointGetSliderPositionRate(mID); }
	
	//
	// params

	void	setLoStopPos( float value ) { setParam(PARAM_LO_STOP,value); }
	void	setHiStopPos( float value ) { setParam(PARAM_HI_STOP,value); }
	void	setStopBounce( float value ) { setParam(PARAM_BOUNCE,value); }
	void	setStopCFM( float value ) { setParam(PARAM_STOP_CFM,value); }
	void	setStopERP( float value ) { setParam(PARAM_STOP_ERP,value); }
	void	setMotorVelocity( float value ) { setParam(PARAM_VEL,value); }
	void	setMotorMaxForce( float value ) { setParam(PARAM_FMAX,value); }
	void	setFudgeFactor( float value ) { setParam(PARAM_FUDGE,value); }
	void	setCFM( float value ) { setParam(PARAM_CFM,value); }

	float	getLoStopPos() const { return getParam(PARAM_LO_STOP); }
	float	getHiStopPos() const { return getParam(PARAM_HI_STOP); }
	float	getStopBounce() const { return getParam(PARAM_BOUNCE); }
	float	getStopCFM() const { return getParam(PARAM_STOP_CFM); }
	float	getStopERP() const { return getParam(PARAM_STOP_ERP); }
	float	getMotorVelocity() const { return getParam(PARAM_VEL); }
	float	getMotorMaxForce() const { return getParam(PARAM_FMAX); }
	float	getFudgeFactor() const { return getParam(PARAM_FUDGE); }
	float	getCFM() const { return getParam(PARAM_CFM); }

private:
	void setParam( eParam param, float value ) { dJointSetSliderParam(mID, param, value); }
	float getParam( eParam param ) const { return (float)dJointGetSliderParam(mID, param); }
};



// --------------------------------------------------------------------------

class CPhysicsJointUniversal : public CPhysicsJoint {
public:
	CPhysicsJointUniversal() { }
	CPhysicsJointUniversal( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) { mID = dJointCreateUniversal (world, group); }
	
	void create( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) {
		if (mID) dJointDestroy (mID);
		mID = dJointCreateUniversal (world, group);
	}
	
	void setAnchor( const SVector3& a ) { dJointSetUniversalAnchor(mID, a.x, a.y, a.z); }
	void setAxis1( const SVector3& a ) { dJointSetUniversalAxis1 (mID, a.x, a.y, a.z); }
	void setAxis2( const SVector3& a ) { dJointSetUniversalAxis2 (mID, a.x, a.y, a.z); }
	
	SVector3 getAnchor() const { dVector3 r; dJointGetUniversalAnchor (mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	SVector3 getAxis1() const { dVector3 r; dJointGetUniversalAxis1 (mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	SVector3 getAxis2() const { dVector3 r; dJointGetUniversalAxis2 (mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
};



// --------------------------------------------------------------------------

class CPhysicsJointHinge2 : public CPhysicsJoint {
public:
	CPhysicsJointHinge2() { }
	CPhysicsJointHinge2( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) { mID = dJointCreateHinge2 (world, group); }
	
	void create( CPhysicsContext& world, TPhysicsJointGroupID group=0 ) {
		if (mID) dJointDestroy (mID);
		mID = dJointCreateHinge2 (world, group);
	}
	
	void setAnchor( const SVector3& a ) { dJointSetHinge2Anchor(mID, a.x, a.y, a.z); }
	void setAxis1( const SVector3& a ) { dJointSetHinge2Axis1 (mID, a.x, a.y, a.z); }
	void setAxis2( const SVector3& a ) { dJointSetHinge2Axis2 (mID, a.x, a.y, a.z); }
	
	SVector3 getAnchor() const { dVector3 r; dJointGetHinge2Anchor (mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	SVector3 getAxis1() const { dVector3 r; dJointGetHinge2Axis1 (mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }
	SVector3 getAxis2() const { dVector3 r; dJointGetHinge2Axis2 (mID, r); return SVector3((float)r[0],(float)r[1],(float)r[2]); }

	float getAngle1() const { return (float)dJointGetHinge2Angle1(mID); }
	float getAngle1Rate() const { return (float)dJointGetHinge2Angle1Rate(mID); }
	float getAngle2Rate() const { return (float)dJointGetHinge2Angle2Rate(mID); }
	
	//
	// params

	void	setLoStopAngle( float value ) { setParam(PARAM_LO_STOP,value); }
	void	setHiStopAngle( float value ) { setParam(PARAM_HI_STOP,value); }
	void	setStopBounce( float value ) { setParam(PARAM_BOUNCE,value); }
	void	setStopCFM( float value ) { setParam(PARAM_STOP_CFM,value); }
	void	setStopERP( float value ) { setParam(PARAM_STOP_ERP,value); }
	void	setMotorVelocity( float value ) { setParam(PARAM_VEL,value); }
	void	setMotorMaxForce( float value ) { setParam(PARAM_FMAX,value); }
	void	setFudgeFactor( float value ) { setParam(PARAM_FUDGE,value); }
	void	setCFM( float value ) { setParam(PARAM_CFM,value); }
	void	setSuspensionCFM( float value ) { setParam(PARAM_SUSPENSION_CFM,value); }
	void	setSuspensionERP( float value ) { setParam(PARAM_SUSPENSION_ERP,value); }

	void	setMotor2Velocity( float value ) { setParam(PARAM_VEL_2,value); }
	void	setMotor2MaxForce( float value ) { setParam(PARAM_FMAX_2,value); }
	void	setCFM2( float value ) { setParam(PARAM_CFM_2,value); }

	float	getLoStopAngle() const { return getParam(PARAM_LO_STOP); }
	float	getHiStopAngle() const { return getParam(PARAM_HI_STOP); }
	float	getStopBounce() const { return getParam(PARAM_BOUNCE); }
	float	getStopCFM() const { return getParam(PARAM_STOP_CFM); }
	float	getStopERP() const { return getParam(PARAM_STOP_ERP); }
	float	getMotorVelocity() const { return getParam(PARAM_VEL); }
	float	getMotorMaxForce() const { return getParam(PARAM_FMAX); }
	float	getFudgeFactor() const { return getParam(PARAM_FUDGE); }
	float	getCFM() const { return getParam(PARAM_CFM); }
	float	getSuspensionCFM() const { return getParam(PARAM_SUSPENSION_CFM); }
	float	getSuspensionERP() const { return getParam(PARAM_SUSPENSION_ERP); }

	float	getMotor2Velocity() const { return getParam(PARAM_VEL_2); }
	float	getMotor2MaxForce() const { return getParam(PARAM_FMAX_2); }
	float	getCFM2() const { return getParam(PARAM_CFM_2); }

private:
	void setParam( eParam param, float value ) { dJointSetHinge2Param(mID, param, value); }
	float getParam( eParam param ) const { return (float)dJointGetHinge2Param(mID, param); }
};


/*
class dFixedJoint : public CPhysicsJoint {
public:
	dFixedJoint() { }
	dFixedJoint (CPhysicsContext& world, TPhysicsJointGroupID group=0)
	{ mID = dJointCreateFixed (world, group); }
	
	void create (CPhysicsContext& world, TPhysicsJointGroupID group=0) {
		if (mID) dJointDestroy (mID);
		mID = dJointCreateFixed (world, group);
	}
	
	void set()
	{ dJointSetFixed (mID); }
};
*/

/*
class CPhysicsJointContact : public CPhysicsJoint {
public:
	CPhysicsJointContact() { }
	CPhysicsJointContact( CPhysicsContext& world, TPhysicsJointGroupID group, dContact *contact ) { mID = dJointCreateContact(world, group, contact); }
	
	void create( CPhysicsContext& world, TPhysicsJointGroupID group, dContact *contact ) {
		if(mID) dJointDestroy (mID);
		mID = dJointCreateContact(world, group, contact);
	}
};
*/


/*
class dAMotorJoint : public CPhysicsJoint {
public:
	dAMotorJoint() { }
	dAMotorJoint (CPhysicsContext& world, TPhysicsJointGroupID group=0)
	{ mID = dJointCreateAMotor (world, group); }
	
	void create (CPhysicsContext& world, TPhysicsJointGroupID group=0) {
		if (mID) dJointDestroy (mID);
		mID = dJointCreateAMotor (world, group);
	}
	
	void setMode (int mode)
	{ dJointSetAMotorMode (mID, mode); }
	int getMode() const
	{ return dJointGetAMotorMode (mID); }
	
	void setNumAxes (int num)
	{ dJointSetAMotorNumAxes (mID, num); }
	int getNumAxes() const
	{ return dJointGetAMotorNumAxes (mID); }
	
	void setAxis (int anum, int rel, float x, float y, float z)
	{ dJointSetAMotorAxis (mID, anum, rel, x, y, z); }
	void getAxis (int anum, dVector3 result) const
	{ dJointGetAMotorAxis (mID, anum, result); }
	int getAxisRel (int anum) const
	{ return dJointGetAMotorAxisRel (mID, anum); }
	
	void setAngle (int anum, float angle)
	{ dJointSetAMotorAngle (mID, anum, angle); }
	float getAngle (int anum) const
	{ return dJointGetAMotorAngle (mID, anum); }
	float getAngleRate (int anum)
	{ return dJointGetAMotorAngleRate (mID,anum); }
	
	void setParam (int parameter, float value)
	{ dJointSetAMotorParam (mID, parameter, value); }
	float getParam (int parameter) const
	{ return dJointGetAMotorParam (mID, parameter); }
};
*/


}; // namespace

#endif
