// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __PHYSICSABLE_ODE_H
#define __PHYSICSABLE_ODE_H


#include <ode/ode.h>
#include "../math/Vector3.h"
#include "../math/Matrix4x4.h"
#include "../math/Quaternion.h"
#include "PhysicsContext.h"


namespace dingus {


struct SPhysicsMass {
public:
	SPhysicsMass() : mass(1.0f) { inertiaTensor.identify(); }
public:
	SMatrix4x4	inertiaTensor;
	float		mass;
};



// --------------------------------------------------------------------------

class CPhysicsable : public boost::noncopyable {
public:
	CPhysicsable() : mID(0) { }
	CPhysicsable( CPhysicsContext& world ) { mID = dBodyCreate(world); }
	~CPhysicsable() { if(mID) dBodyDestroy(mID); } // no virtual intentionally
	
	void create( CPhysicsContext& world ) {
		if(mID) dBodyDestroy (mID);
		mID = dBodyCreate(world);
	}
	
	TPhysicsableID getID() const { return mID; }
	operator TPhysicsableID() const { return mID; }
	
	void setData( void *data ) { dBodySetData(mID, data); }
	void *getData() const { return dBodyGetData(mID); }
	
	void setPosition( const SVector3& pos ) { dBodySetPosition(mID,pos.x,pos.y,pos.z); }
	void setRotation( const SMatrix4x4& rot );
	//void setRotation( const SQuaternion& q ) { dBodySetQuaternion(mID,q); }
	void setLinearVel( const SVector3& v ) { dBodySetLinearVel(mID,v.x,v.y,v.z); }
	void setAngularVel( const SVector3& v ) { dBodySetAngularVel(mID,v.x,v.y,v.z); }
	
	SVector3 getPosition() const { const dReal* v = dBodyGetPosition(mID); return SVector3((float)v[0],(float)v[1],(float)v[2]); }
	//SQuaternion getRotation() const { return SQuaternion( dBodyGetQuaternion(mID) ); }
	void getRotation( SMatrix4x4& dest ) const;
	SVector3 getLinearVel() const { const dReal* v = dBodyGetLinearVel(mID); return SVector3((float)v[0],(float)v[1],(float)v[2]); }
	SVector3 getAngularVel() const { const dReal* v = dBodyGetAngularVel(mID); return SVector3((float)v[0],(float)v[1],(float)v[2]); }

	void	setMass( const SPhysicsMass& mass );
	
	void	addForce( const SVector3& f ) { dBodyAddForce(mID, f.x, f.y, f.z); }
	void	addTorque( const SVector3& t ) { dBodyAddTorque(mID, t.x, t.y, t.z); }
	void	addLocalForce( const SVector3& f ) { dBodyAddRelForce(mID, f.x, f.y, f.z); }
	void	addLocalTorque( const SVector3& t ) { dBodyAddRelTorque(mID, t.x, t.y, t.z); }
	void	addForceAtPos( const SVector3& f, const SVector3& pos ) { dBodyAddForceAtPos(mID, f.x, f.y, f.z, pos.x, pos.y, pos.z); }
	void	addForceAtLocalPos( const SVector3& f, const SVector3& pos ) { dBodyAddForceAtRelPos(mID, f.x, f.y, f.z, pos.x, pos.y, pos.z); }
	void	addLocalForceAtPos( const SVector3& f, const SVector3& pos ) { dBodyAddRelForceAtPos(mID, f.x, f.y, f.z, pos.x, pos.y, pos.z); }
	void	addLocalForceAtLocalPos( const SVector3& f, const SVector3& pos ) { dBodyAddRelForceAtRelPos(mID, f.x, f.y, f.z, pos.x, pos.y, pos.z); }
	
	SVector3 getForce() const { const dReal* v = dBodyGetForce(mID); return SVector3((float)v[0],(float)v[1],(float)v[2]); }
	SVector3 getTorque() const { const dReal* v = dBodyGetTorque(mID); return SVector3((float)v[0],(float)v[1],(float)v[2]); }
	void	setForce( const SVector3& f ) { dBodySetForce(mID,f.x,f.y,f.z); }
	void	setTorque( const SVector3& t ) { dBodySetTorque(mID,t.x,t.y,t.z); }
	
	void	enable() { dBodyEnable (mID); }
	void	disable() { dBodyDisable (mID); }
	bool	isEnabled() const { return dBodyIsEnabled(mID)?true:false; }

	//void	setAutoDisable( bool ad ) { dBodySetAutoDisableSF1(mID,ad); }
	
	///** Given local point, gets it's world position. */
	//void	getLocalPtPos( const SVector3& p, SVector3& res ) const { dBodyGetRelPointPos(mID, p.x,p.y,p.z, res); }
	///** Given local point, gets it's world velocity. */
	//void	getLocalPtVel( const SVector3& p, SVector3& res ) const { dBodyGetRelPointVel(mID, p.x,p.y,p.z, res); }
	///** Given world point, gets it's world velocity. */
	//void	getPtVel( const SVector3& p, SVector3& res ) const { dBodyGetPointVel(mID, p.x,p.y,p.z, res); }
	///** Given world point, gets it's local position. */
	//void	getPtLocalPos( const SVector3& p, SVector3& res ) const { dBodyGetPosRelPoint(mID, p.x,p.y,p.z, res); }
	///** Rotates local vector into world. */
	//void	vectorToWorld( const SVector3& p, SVector3& res ) const { dBodyVectorToWorld(mID, p.x,p.y,p.z, res); }
	///** Rotates world vector into local. */
	//void	vectorFromWorld( const SVector3& p, SVector3& res ) const { dBodyVectorFromWorld(mID, p.x,p.y,p.z, res); }
	
	
	void setFiniteRotationMode( bool finite ) { dBodySetFiniteRotationMode(mID, finite?1:0); }
	bool isFiniteRotationMode() const { return dBodyGetFiniteRotationMode(mID)?true:false; }
	void setFiniteRotationAxis( const SVector3& a ) { dBodySetFiniteRotationAxis (mID, a.x, a.y, a.z); }
	void getFiniteRotationAxis( SVector3& res ) const { dVector3 r; dBodyGetFiniteRotationAxis(mID, r); res.x=(float)r[0];res.y=(float)r[1];res.z=(float)r[2]; }
	
	int getJointCount() const { return dBodyGetNumJoints(mID); }
	//TJointID getJoint(int index) const { return dBodyGetJoint(mID, index); }
	
	void setGravitable( bool g ) { dBodySetGravityMode(mID,g?1:0); }
	bool isGravitable() const { return dBodyGetGravityMode(mID)?true:false; }
	
	bool isConnectedTo( TPhysicsableID body ) const { return dAreConnected(mID, body)?true:false; }

private:
	TPhysicsableID mID;
};


}; // namespace

#endif
