// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __EFFECTSLOT_H
#define __EFFECTSLOT_H

#include "../kernel/Proxies.h"

namespace dingus {


class CRenderContext;
class CRenderable;
class CResourceId;


/**
 *  Groups renderables (CRenderable) by their effect.
 *
 *  Has a CD3DXEffect and a bunch of renderables to render with it. Note
 *  that the order in which renderables are processed isn't determined.
 *  Effect slots are containen in CRenderLayer.
 */
class CEffectSlot {
protected:
	CEffectSlot( CD3DXEffect& fx );
	
public:
	static CEffectSlot* createEffectSlot( CResourceId const& id );

	/**
	 *  Add new renderable to the slot.
	 *  Note that currently the renderable can't be contained in several effect
	 *  slots due to the way effect params are cached.
	 */
	void addRenderable( CRenderable& r ) { mRenderables.push_back( &r ); }

	/**
	 *  Removes the renderable from the slot.
	 *  This can be costly operation - it's O(N), where N is number of renderables
	 *  in this slot.
	 */
	void removeRenderable( CRenderable& r ) { mRenderables.remove( &r ); }

	bool isEmpty() { return mRenderables.empty(); }

	/**
	 *  Removes all renderables.
	 */
	void clear() { mRenderables.clear(); }


	/**
	 *  Render the effect slot.
	 */
	void render( CRenderContext& ctx );

	/**
	 *  Called by the framework whenever the effect is reloaded. The proxy
	 *  object CD3DXEffect remains the same, but the real D3DX effect may have
	 *  changed. This re-sets the effect on all current renderables' params, so
	 *  they can initialize themselves. Note that for any new renderables added
	 *  by addRenderable() the current effect is set automatically.
	 */
	void notifyChanged();

	/**
	 *  Find and set a valid effect technique.
	 */
	void validate();

	CD3DXEffect& getEffect() const { return mEffect; }
	
private:
	typedef fastvector<CRenderable*> TRenderableFVector;

private:
	TRenderableFVector	mRenderables;
	CD3DXEffect&		mEffect;
};


};

#endif
