// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __RENDER_CONTEXT_H
#define __RENDER_CONTEXT_H


#include "RenderCamera.h"
#include "RenderLayer.h"
#include "RenderStats.h"
#include "../kernel/FrameNumber.h"
#include "../utils/AbstractNotifier.h"
#include "../kernel/Proxies.h"
#include "EffectParams.h"


namespace dingus {


/**
 *  Gets notified when scene rendering occurs.
 */
class IRenderContextListener {
public:
	virtual ~IRenderContextListener() = 0 {};
	
	virtual void beforeRendering( CRenderContext& ctx ) = 0;
	virtual void afterRendering( CRenderContext& ctx ) = 0;
};



class CRenderContext {
public:
	enum { LAYER_COUNT = 10 };
	enum { DEFAULT_LAYER = 5 };

public:
	CRenderContext();
	~CRenderContext();

	const CRenderLayer& getLayer( int num ) const { return *mLayers[num]; }
	CRenderLayer& getLayer( int num ) { return *mLayers[num]; }
	
	const CRenderCamera& getCamera() const { return mRenderCamera; };
	CRenderCamera& getCamera() { return mRenderCamera; };

	void render();
	
	CFrameNumber const& getCurrentFrame() const { return mCurrentFrame; };
	CFrameNumber& getCurrentFrame() { return mCurrentFrame; };

	CRenderStats& getStats() const { return mStats; }

	void addListener( IRenderContextListener& l ) { mRenderListeners.addListener(l); }
	void removeListener( IRenderContextListener& l ) { mRenderListeners.removeListener(l); }

	/** Called by the framework whenever the effects are reloaded. */
	void notifyChanged();

	CEffectParams const& getGlobalParams() const { return mGlobalParams; };
	CEffectParams& getGlobalParams() { return mGlobalParams; };
	
private:
	class CRenderNotifier : public CSimpleNotifier<IRenderContextListener> {
	public:
		void beforeRendering( CRenderContext& ctx ) {
			TListenerVector::iterator it, itEnd = getListeners().end();
			for( it = getListeners().begin(); it != getListeners().end(); ++it ) {
				assert( *it );
				(*it)->beforeRendering( ctx );
			}
		}
		void afterRendering( CRenderContext& ctx ) {
			TListenerVector::iterator it, itEnd = getListeners().end();
			for( it = getListeners().begin(); it != getListeners().end(); ++it ) {
				assert( *it );
				(*it)->afterRendering( ctx );
			}
		}
	};

	void beforeRendering() { mRenderListeners.beforeRendering( *this ); }
	void afterRendering() { mRenderListeners.afterRendering( *this ); }

private:
	CRenderLayer*	mLayers[LAYER_COUNT];
	CRenderCamera	mRenderCamera;

	CFrameNumber	mCurrentFrame;
	CEffectParams	mGlobalParams;
	CD3DXEffect*	mGlobalEffect;

	CRenderNotifier	mRenderListeners;

	mutable CRenderStats mStats;
	bool			mGlobalInited;
};


}; // namespace

#endif
