// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __RENDER_LAYER_H
#define __RENDER_LAYER_H

#include "../resource/ResourceId.h"
#include "EffectSlot.h"


namespace dingus {


class CRenderContext;
class CRenderable;

/**
 *  A rendering layer.
 *
 *  Render layers enable to order the geometry to be rendered. Eg. transparent
 *  geometry needs to be renderer after the solid geometry. Each render layer
 *  contains a bunch of effect slots (CEffectSlot). Layers are contained in a
 *  CRenderContext which processes them sequentially.
 */
class CRenderLayer {
public:
	CRenderLayer();
	~CRenderLayer();

	/**
	 *  Add renderable with effect.
	 *  Adds to corresponding effect slot, or creates new slot and adds to it.
	 */
	void addRenderable( CRenderable& r, CResourceId const& fxID ) { getEffect( fxID ).addRenderable( r ); }
	/**
	 *  Remove renderable from effect.
	 */
	void removeRenderable( CRenderable& r, CResourceId const& fxID );
	
	void render( CRenderContext& ctx );
	void clear();

	/** Called by the framework whenever the effects are reloaded. */
	void notifyChanged();

private:
	typedef std::map<CResourceId,CEffectSlot*>	TSlotMap;
	typedef fastvector<CEffectSlot*>			TSlotFVector;

private:
	TSlotMap const& getEffectsMap() const { return mEffectsMap; }
	TSlotMap& getEffectsMap() { return mEffectsMap; }
	
	TSlotFVector const& getEffects() const { return mEffects; }
	TSlotFVector& getEffects() { return mEffects; }

	CEffectSlot& getEffect( CResourceId const& fxID );
	void removeEffect( CResourceId const& fxID );
	
private:
	TSlotMap		mEffectsMap;
	TSlotFVector	mEffects;
};


}; // namespace


#endif
