// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __RENDERABLE_H
#define __RENDERABLE_H


#include "EffectParams.h"
#include "../kernel/FrameNumber.h"
#include "../utils/AbstractNotifier.h"
#include "RenderContext.h"


namespace dingus {


class CEffectSlot;
class CRenderable;


class IRenderListener {
public:
	virtual ~IRenderListener() = 0 {};

	virtual void beforeRender( CRenderable& r, CEffectSlot const& fx ) = 0;
	virtual void afterRender( CRenderable& r, CEffectSlot const& fx ) = 0;
};


class CRenderListenerAdapter : public IRenderListener {
public:
	virtual void beforeRender( CRenderable& r, CEffectSlot const& fx ) { };
	virtual void afterRender( CRenderable& r, CEffectSlot const& fx ) { };
};


/**
 *  Base renderable class.
 *
 *  Contains parameters for the rendering effect (mParams), virtual method renderContent()
 *  and before/after rendering listeners. Note that renderContent() may be called multiple
 *  times if the effect is multi-pass.
 *
 *  Renderables are contained in CEffectSlot, which groups renderables by their
 *  effect.
 *
 *  The renderable should generally only render it's geometry. The render states
 *  are to be managed by the effect and corresponding params.
 */
class CRenderable : public CSimpleNotifier<IRenderListener> {
public:
	CRenderable() { };
	virtual ~CRenderable() { };

	CEffectParams const& getParams() const { return mParams; }
	CEffectParams& getParams() { return mParams; }

	/** Notifies render listeners that the renderable will be rendered. */
	void beforeRender( CRenderContext const& ctx, CEffectSlot const& fx );
	/** Notifies render listeners that the renderable was just rendered. */
	void afterRender( CRenderContext const& ctx, CEffectSlot const& fx );

	/** Called for each rendering pass. If renderable is active, calls renderContent(). */
	void render( CRenderContext const& ctx, CEffectSlot const& fx ) {
		/*if( isActive(ctx) )*/ renderContent( ctx, fx );
	}

	// TBD: must be refined
	void setActive( CRenderContext const& ctx, bool active = true ) {
		mActiveOnFrame = ctx.getCurrentFrame();
		if( !active )
			mActiveOnFrame.increase(-1); 
	}
	bool isActive( CRenderContext const& ctx ) const { return ctx.getCurrentFrame() == mActiveOnFrame; }

protected:
	/**
	 *  Called for each rendering pass. Here one should render it's geometry.
	 *  @param ctx The render context that is rendering.
	 *  @param fx The effect slot that is rendering.
	 */
	virtual void renderContent( CRenderContext const& ctx, CEffectSlot const& fx ) { };

private:
	CEffectParams	mParams;
	CFrameNumber	mActiveOnFrame;
};


};

#endif
