// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "RenderableBuffer.h"
#include "RenderContext.h"
#include "../kernel/D3DDevice.h"

using namespace dingus;



// --------------------------------------------------------------------------
//  CAbstractRenderableBuffer
// --------------------------------------------------------------------------

CAbstractRenderableBuffer::CAbstractRenderableBuffer()
:	mPrimType( D3DPT_TRIANGLELIST ),
	mPrimCount( 0 )
{
	mLargestActiveStream = VERTEX_STREAM_COUNT - 1;
	resetVBs();
	mLargestActiveStream = -1;
}

void CAbstractRenderableBuffer::resetVBs()
{
	for( int q = 0; q <= mLargestActiveStream; ++q ) {
		mVB[q] = NULL;
		mStride[q] = 0;
		mByteOffset[q] = 0;
	}
	mLargestActiveStream = -1;
	mVertexDecl = NULL;
}

void CAbstractRenderableBuffer::setVB( CD3DVertexBuffer& vb, int stream )
{
	assert( stream >= 0 && stream < VERTEX_STREAM_COUNT );
	mVB[stream] = &vb;
	if( stream > mLargestActiveStream )
		mLargestActiveStream = stream;
}

void CAbstractRenderableBuffer::applyStreams()
{
	IDirect3DDevice9& dx = *gD3DDevice;

	for( int q = 0; q <= mLargestActiveStream; ++q ) {
		assert( mStride[q] > 0 );

		dx.SetStreamSource( q, mVB[q]->getObject(), mByteOffset[q], mStride[q] );
	}

	if( mVertexDecl ) {
		dx.SetVertexDeclaration( mVertexDecl->getObject() );
	}
}

void CAbstractRenderableBuffer::unapplyStreams()
{
	IDirect3DDevice9& dx = *gD3DDevice;

	for( int q = 0; q <= mLargestActiveStream; ++q ) {
		dx.SetStreamSource( q, NULL, 0, 0 );
	}
}


// --------------------------------------------------------------------------
//  CRenderableBuffer
// --------------------------------------------------------------------------

CRenderableBuffer::CRenderableBuffer()
:	mStartVertex( 0 )
{
}


void CRenderableBuffer::renderContent( CRenderContext const& ctx, CEffectSlot const& fx )
{
	if( getLargestActiveStream() < 0 )
		return;
	if( getPrimCount() == 0 )
		return;
	applyStreams();
	gD3DDevice->DrawPrimitive( getPrimType(), mStartVertex, getPrimCount() );
	unapplyStreams();
	//stats
	ctx.getStats().incDrawCalls();
	ctx.getStats().incVerticesRendered( /* TBD */ getPrimCount() );
	ctx.getStats().incPrimsRendered( getPrimCount() );
}


// --------------------------------------------------------------------------
//  CRenderableIndexedBuffer
// --------------------------------------------------------------------------

CRenderableIndexedBuffer::CRenderableIndexedBuffer()
:	mIB( NULL ),
	mBaseVertex( 0 ),
	mMinVertex( 0 ),
	mNumVertices( 0 ),
	mStartIndex( 0 )
{
}

void CRenderableIndexedBuffer::renderContent( CRenderContext const& ctx, CEffectSlot const& fx )
{
	if( getLargestActiveStream() < 0 )
		return;
	if( getPrimCount() == 0 )
		return;

	assert( mIB );

	applyStreams();

	gD3DDevice->SetIndices( mIB->getObject() );

	gD3DDevice->DrawIndexedPrimitive(
		getPrimType(), mBaseVertex, mMinVertex, mNumVertices, mStartIndex, getPrimCount() );

	unapplyStreams();
	gD3DDevice->SetIndices( NULL );

	// stats
	ctx.getStats().incDrawCalls();
	ctx.getStats().incVerticesRendered( mNumVertices );
	ctx.getStats().incPrimsRendered( getPrimCount() );
}
