// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __RENDERABLE_BUFFER_H
#define __RENDERABLE_BUFFER_H

#include "Renderable.h"
#include "../kernel/Proxies.h"


namespace dingus {


class CRenderContext;


// --------------------------------------------------------------------------

/**
 *  Abstract base class for vertex and indexed vertex buffer renderables.
 */
class CAbstractRenderableBuffer : public CRenderable {
public:
	enum { VERTEX_STREAM_COUNT = 16 };
public:
	CAbstractRenderableBuffer();

	void setVB( CD3DVertexBuffer& vb, int stream = 0 );
	CD3DVertexBuffer const* getVB( int stream = 0 ) const { return mVB[stream]; }
	CD3DVertexBuffer* getVB( int stream = 0 ) { return mVB[stream]; }

	/** Resets VBs, strides, byte offsets and vertex decl. */
	void resetVBs();

	/** Set stream element size in bytes. */
	void setStride( int stride, int stream = 0 ) { mStride[stream] = stride; }
	/** Get stream element size in bytes. */
	int getStride( int stream = 0 ) const { return mStride[stream]; }

	/** Set stream offset in bytes. Can be not supported on pre-DX9 drivers. */
	void setByteOffset( int offset, int stream = 0 ) { mByteOffset[stream] = offset; }
	/** Get stream offset in bytes. */
	int getByteOffset( int stream = 0 ) const { return mByteOffset[stream]; }

	/** Set vertex declaration. If isn't set, you have to set FVF on the device. */
	void setVertexDecl( CD3DVertexDecl* decl ) { mVertexDecl = decl; }
	/** Get vertex declaration. */
	CD3DVertexDecl* getVertexDecl() const { return mVertexDecl; }

	/** Set primitive type to render. */
	void setPrimType( D3DPRIMITIVETYPE primitiveType ) { mPrimType = primitiveType; }
	/** Get primitive type to render. */
	D3DPRIMITIVETYPE getPrimType() const { return mPrimType; }

	/** Set primitive count to render. */
	void setPrimCount( int primitiveCount ) { mPrimCount = primitiveCount; }
	/** Get primitive count to render. */
	int getPrimCount() const { return mPrimCount; }

protected:
	void	applyStreams();
	void	unapplyStreams();
	int		getLargestActiveStream() const { return mLargestActiveStream; }
	
private:
	CD3DVertexBuffer*	mVB[VERTEX_STREAM_COUNT];
	int					mStride[VERTEX_STREAM_COUNT];
	int					mByteOffset[VERTEX_STREAM_COUNT];
	int					mLargestActiveStream;
	CD3DVertexDecl*		mVertexDecl;

	D3DPRIMITIVETYPE	mPrimType;
	int					mPrimCount;
};


// --------------------------------------------------------------------------


class CRenderableBuffer : public CAbstractRenderableBuffer {
public:
	CRenderableBuffer();

	/** Set vertex index to start rendering from. */
	void setStartVertex( int startVertex ) { mStartVertex = startVertex; }
	/** Get vertex index to start rendering from. */
	int getStartVertex() const { return mStartVertex; }
	
	virtual void renderContent( CRenderContext const& ctx, CEffectSlot const& fx );
	
private:
	int	mStartVertex;
};


// --------------------------------------------------------------------------


class CRenderableIndexedBuffer : public CAbstractRenderableBuffer {
public:
	CRenderableIndexedBuffer();

	void setIB( CD3DIndexBuffer& ibuffer ) { mIB = &ibuffer; }
	const CD3DIndexBuffer* getIB() const { return mIB; }
	CD3DIndexBuffer* getIB() { return mIB; }

	/**
	 *  Set vertex that is treated as the first one. All indices will be relative
	 *  to this vertex. This can be done with setByteOffset() in many cases, but
	 *  sometimes it isn't supported.
	 */
	void setBaseVertex( int baseVertex ) { mBaseVertex = baseVertex; }
	/** Get vertex that is treated as the first one. */
	int getBaseVertex() const { return mBaseVertex; }

	/** Set the lowest vertex that will be referenced by IB. */
	void setMinVertex( int minVertex ) { mMinVertex = minVertex; }
	/** Get the lowest vertex that will be referenced by IB. */
	int getMinVertex() const { return mMinVertex; }

	/** Set the number of vertices from min vertex that will be referenced by IB. */
	void setNumVertices( int numVertices ) { mNumVertices = numVertices; }
	/** Get the number of vertices from min vertex that will be referenced by IB. */
	int getNumVertices() const { return mNumVertices; }

	/** Set the index from which to start rendering. */ 
	void setStartIndex( int startIndex ) { mStartIndex = startIndex; }
	/** Get the index from which to start rendering. */ 
	int getStartIndex() const { return mStartIndex; }

	virtual void renderContent( CRenderContext const& ctx, CEffectSlot const& fx );
	
private:
	CD3DIndexBuffer*	mIB;
	int		mBaseVertex;
	int		mMinVertex;
	int		mNumVertices;
	int		mStartIndex;
};



}; // namespace

#endif
