// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "CollisionMeshBundle.h"
#include "../utils/Errors.h"
#include "../kernel/D3DDevice.h"
#include "../collider-ode/InertiaTensorBuilder.h"

using namespace dingus;


void CCollisionMeshBundle::loadMesh( CResourceId const& id, CCollisionMesh& dest ) const
{
	ID3DXBuffer* adjancency = NULL;
	ID3DXBuffer* material = NULL;
	ID3DXBuffer* effects = NULL;
	DWORD matCount;
	ID3DXMesh* mesh = NULL;

	HRESULT hres = D3DXLoadMeshFromX(
		(mPreDir + id.getUniqueName()).c_str(),
		D3DXMESH_SYSTEMMEM,
		gD3DDevice,
		&adjancency,
		&material,
		&effects,
		&matCount,
		&mesh );
	if( !SUCCEEDED( hres ) ) {
		std::string msg = "failed to load mesh '" + id.getUniqueName() + "'";
		CConsole::CON_ERROR.write(msg);
		THROW_DXERROR( hres, msg );
	}
	assert( mesh );

	// release stuff
	if( adjancency )
		adjancency->Release();
	if( material )
		material->Release();
	if( effects )
		effects->Release();

	//
	// build collision mesh

	int i;
	dest.begin( mesh->GetNumVertices(), mesh->GetNumFaces() );

	// VB
	const BYTE* vb;
	mesh->LockVertexBuffer( D3DLOCK_READONLY, (void**)&vb );
	for( i = 0; i < dest.getNumVertices(); ++i ) {
		dest.setVertex( i, *(const SVector3*)vb );
		vb += mesh->GetNumBytesPerVertex();
	}
	mesh->UnlockVertexBuffer();

	// IB
	const short* ib;
	mesh->LockIndexBuffer( D3DLOCK_READONLY, (void**)&ib );
	for( i = 0; i < dest.getNumIndices()/3; ++i ) {
		dest.setTriangle( i, ib[0], ib[1], ib[2] );
		ib += 3;
	}
	mesh->UnlockIndexBuffer();

	mesh->Release();

	// finalize build (build optimized BV tree, etc.)
	dest.end();

	// inertia tensor
	CInertiaTensorBuilder::compute( dest );

	CONSOLE.write( "collision mesh loaded '" + id.getUniqueName() + "'" );
}

CCollisionMesh* CCollisionMeshBundle::loadResourceById( CResourceId const& id )
{
	CCollisionMesh* mesh = new CCollisionMesh();
	loadMesh( id, *mesh );
	return mesh;
}

void CCollisionMeshBundle::clearResource( CCollisionMesh& resource )
{
	delete &resource;
}

void CCollisionMeshBundle::reload()
{
	// reload all objects
	TResourceMap::iterator it;
	for( it = mResourceMap.begin(); it != mResourceMap.end(); ++it ) {
		CCollisionMesh& res = *reinterpret_cast<CCollisionMesh*>(it->second);
		loadMesh( it->first, res );
	}
}

