// --------------------------------------------------------------------------
// Dingus project - a collection of subsystems for game/graphics applications
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "EffectBundle.h"
#include "../utils/Errors.h"
#include "../kernel/D3DDevice.h"

using namespace dingus;



ID3DXEffect* CEffectBundle::loadEffect( CResourceId const& id ) const
{
	ID3DXEffect* fx = NULL;
	ID3DXBuffer* errors = NULL;

	assert( mSharedPool );
	HRESULT hres = D3DXCreateEffectFromFile(
		gD3DDevice,
		(mPreDir + id.getUniqueName()).c_str(),
		NULL, // TBD ==> macros
		NULL, // TBD ==> includes
		0,
		mSharedPool,
		&fx,
		&errors );
	if( errors && errors->GetBufferSize() > 1 ) {
		std::string msg = "failed to compile effect '" + id.getUniqueName() + "'";
		msg += (const char*)errors->GetBufferPointer();
		CConsole::CON_ERROR.write( msg );
	}


	if( FAILED( hres ) ) {
		std::string msg = "failed to compile effect '" + id.getUniqueName() + "'";
		CConsole::CON_ERROR.write( msg );
		THROW_DXERROR( hres, msg );
	}

	assert( fx );
	CONSOLE.write( "fx loaded '" + id.getUniqueName() + "'" );

	if( errors )
		errors->Release();
	
	return fx;
}

CD3DXEffect* CEffectBundle::loadResourceById( CResourceId const& id )
{
	ID3DXEffect* fx = loadEffect( id );
	return new CD3DXEffect( fx );
}

void CEffectBundle::clearResource( CD3DXEffect& resource )
{
	ULONG refs = resource.getObject()->Release();
}


void CEffectBundle::createResource()
{
	// create pool
	assert( !mSharedPool );
	D3DXCreateEffectPool( &mSharedPool );
	assert( mSharedPool );

	// reload all objects
	TResourceMap::iterator it;
	for( it = mResourceMap.begin(); it != mResourceMap.end(); ++it ) {
		CD3DXEffect& res = *reinterpret_cast<CD3DXEffect*>(it->second);
		assert( res.isNull() );
		res.setObject( loadEffect( it->first ) );
		assert( !res.isNull() );
	}
}

void CEffectBundle::activateResource()
{
	// call reset on effects
	TResourceMap::iterator it;
	for( it = mResourceMap.begin(); it != mResourceMap.end(); ++it ) {
		CD3DXEffect& res = *reinterpret_cast<CD3DXEffect*>(it->second);
		assert( !res.isNull() );
		res.getObject()->OnResetDevice();
	}
}

void CEffectBundle::passivateResource()
{
	// call lost on effects
	TResourceMap::iterator it;
	for( it = mResourceMap.begin(); it != mResourceMap.end(); ++it ) {
		CD3DXEffect& res = *reinterpret_cast<CD3DXEffect*>(it->second);
		assert( !res.isNull() );
		res.getObject()->OnLostDevice();
	}
}

void CEffectBundle::deleteResource()
{
	// unload all objects
	TResourceMap::iterator it;
	for( it = mResourceMap.begin(); it != mResourceMap.end(); ++it ) {
		CD3DXEffect& res = *reinterpret_cast<CD3DXEffect*>(it->second);
		assert( !res.isNull() );
		clearResource( res );
		res.setObject( NULL );
	}

	// release pool
	assert( mSharedPool );
	mSharedPool->Release();
	mSharedPool = NULL;
}

