#include "stdafx.h"
#pragma hdrstop

#include "Jammer.h"
#include "jammer/JammerStaticEntity.h"
#include "jammer/JammerThingEntity.h"
#include <dingus/kernel/SystemClock.h>
#include <dingus/collider-ode/CollisionContext.h>
#include <unco/kernel/Contexts.h>
#include <unco/kernel/AppContext.h>
#include <jam/level/Level.h>
#include <jam/level/GameInfo.h>
#include <jam/entity/CollideClasses.h>
#include <jam/entity/EntityClasses.h>

CJammer::CJammer()
:	mEditMode(MODE_CREATE),
	mSelectedStatic(0),
	mSelectedThing(0),
	mSelectedTag(0),
	mSelectedTagForLink(0),
	mCamera(NULL),
	mJamLevel(NULL),
	mLinkFrameCounter(-1),
	mPicker( NULL, 1000.0f ),
	mPickerX(0), mPickerY(0),
	mLastPickDistance(1.0e6f),
	mPickStatic(1<<COL_ENTITY_STATIC),
	mPickThing(1<<COL_ENTITY_THING),
	mPickTag(1<<COL_UI_TAGS)
{
	//
	// vis
	//unco::CContexts::getVisCtx().addDB( CLevel::DB_OVERLAY, *(new dingus::CDummyVisibilityDB()) );

	mRenderTarget.setClearValues( true, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, 0xFF404040, 1.0f, 0 );
}

CJammer::~CJammer()
{
}

void CJammer::initialize()
{
	assert( !mJamLevel );
	mJamLevel = new CJammerLevel();

	assert( !mCamera );
	mCamera = new unco::CCamera( mRenderTarget );
	mCameraController.setAffectedEntity( *mCamera );
	mJamLevel->getStuff().addEntity( mCamera );
	mCamera->attach();
	mCamera->getMatrix().getOrigin().z = -80.0f;
	mCameraController.arcBallSetDistance( 80.0f );

	mPicker.setCategoryBits( 1<<COL_UI_PICKER );
	mPicker.setCollideBits( (1<<COL_ENTITY_STATIC) | (1<<COL_ENTITY_THING) | (1<<COL_UI_TAGS) );
	mPicker.setUserData( this );
	unco::CContexts::getCollisionCtx().addCollidable( mPicker );

	mPicker.addListener( mPickStatic );
	mPicker.addListener( mPickThing );
	mPicker.addListener( mPickTag );
}

void CJammer::setEditMode( eEditMode m )
{
	mEditMode = m;

	// create mode stuff
	bool createEnabled = m == MODE_CREATE;
	mPickStatic.setActive( createEnabled );
	mPickThing.setActive( createEnabled );

	// link mode stuff
	bool linkEnabled = m == MODE_LINK;
	mPickTag.setActive( linkEnabled );
}

CJammerStaticEntity& CJammer::addStaticEntity( const std::string& className )
{
	CLevel& level = CGameInfo::getInstance().getLevel();
	CStaticEntityClass const* clazz = CGameInfo::getInstance().getClasses().getStaticEntityClass( className );
	assert( clazz );
	std::string name = mNameGenerator.generateName(className);
	CJammerStaticEntity* e = new CJammerStaticEntity( name, *clazz, *mJamLevel, 0, level.getVisStatic() );
	level.getStaticEntities().addEntity( e );
	SMatrix4x4 pickRot;
	mPicker.getRotation( pickRot );
	e->getMatrix().getOrigin() = mPicker.getPosition() + pickRot.getAxisZ() * 80.0f;
	e->attach();
	return *e;
}

CJammerThingEntity* CJammer::addThingEntity( const std::string& className )
{
	if( mSelectedStatic ) {
		CLevel& level = CGameInfo::getInstance().getLevel();
		CEntityClass const* clazz = CGameInfo::getInstance().getClasses().getThingEntityClass( className );
		assert( clazz );
		std::string name = mNameGenerator.generateName(className);
		CJammerThingEntity* e = new CJammerThingEntity( name, *clazz, 0, level.getVisThing() );
		e->setParent( mSelectedStatic );
		level.getThingEntities().addEntity( e );
		e->attach();

		SMatrix4x4 m;
		m.identify();
		m.getAxisY() = mPickStatic.getMinNormal();
		if( mSelectedStatic->getCullMode() == D3DCULL_CW )
			m.getAxisY() = -m.getAxisY();
		m.spaceFromAxisY();
		m.getOrigin() = mPickStatic.getMinPosition();

		SMatrix4x4 invPar = mSelectedStatic->getWorldMatrix();
		invPar.invert();

		e->getMatrix() = m * invPar;

		return e;
	} else {
		return NULL;
	}
}

void CJammer::removeStaticEntity( CJammerStaticEntity& entity )
{
	entity.detach();
	CLevel& level = CGameInfo::getInstance().getLevel();
	level.getStaticEntities().removeEntity( &entity );
	if( mSelectedStatic == &entity ) {
		mSelectedStatic = NULL;
		mPickStatic.reset();
	}

	// remove thing entities with this parent
	CLevel::TBaseEntityContainer::TEntityMap::iterator it;
	for( it = level.getThingEntities().getEntities().begin(); it != level.getThingEntities().getEntities().end(); /* */ ) {
		unco::TEntityPtr e = it->second;
		if( &e->getParent() == &entity ) {
			e->detach();
			it = level.getThingEntities().getEntities().erase( it );
		} else {
			++it;
		}
	}
}

void CJammer::removeThingEntity( CJammerThingEntity& entity )
{
	entity.detach();
	CLevel& level = CGameInfo::getInstance().getLevel();
	level.getThingEntities().removeEntity( &entity );
	if( mSelectedThing == &entity ) {
		mSelectedThing = NULL;
		mPickThing.reset();
	}
}

void CJammer::select()
{
	if( mEditMode == MODE_LINK ) {
		if( mSelectedTagForLink && !mSelectedTagForLink->isLinked() ) {
			if( mSelectedTag && !mSelectedTag->isLinked() ) {
				if( mSelectedTag != mSelectedTagForLink && mSelectedTag->getOwner()!=mSelectedTagForLink->getOwner() ) {
					incLinkFrameCounter();
					linkStaticTags( *mSelectedTagForLink, *mSelectedTag );
				}
			}
		}
		mSelectedTagForLink = mSelectedTag;
	}
}

void CJammer::flipStaticLink( SVector3 const& flip ) const
{
	if( mSelectedTagForLink ) {
		SVector3 f = mSelectedTagForLink->getFlips();
		f.x *= flip.x;
		f.y *= flip.y;
		f.z *= flip.z;
		mSelectedTagForLink->setFlips( f );
		if( mSelectedTagForLink->isLinked() ) {
			linkStaticTags( *(CJammerTagInstance*)mSelectedTagForLink->getLink(), *mSelectedTagForLink );
		}
	}
}

void CJammer::moveStaticEntity( CJammerStaticEntity& entity ) const
{
	entity.update(); // recalc tags :)

	CJammerStaticEntity::TTagInstanceVector::iterator it, itEnd = entity.getTagInstances().end();
	for( it = entity.getTagInstances().begin(); it != itEnd; ++it ) {
		CJammerTagInstance& tag = *it;
		if( !tag.isLinked() || tag.getVisitedOnFrame() == mLinkFrameCounter )
			continue;
		linkStaticTags( tag, *(CJammerTagInstance*)tag.getLink() );
	}
}


void CJammer::perform()
{
	int frame = dingus::CSystemClock::getInstance().getPerformCount();
	incLinkFrameCounter();

	mCamera->setProjectionParams(
		0.30f * 3.1415927f,
		unco::CAppContext::getInstance().getWindowAspect(),
		1.0f, 500.0f
	);

	//
	// picking

	SVector3 pickerDir = mCamera->getWorldRay( mPickerX, mPickerY );
	mPicker.setParams( mCamera->getWorldMatrix().getOrigin(), pickerDir );

	// pick statics
	mSelectedStatic = (CJammerStaticEntity*)mPickStatic.getMinColData();
	if( mPickThing.getMinDistance() < mPickStatic.getMinDistance() )
		mSelectedStatic = NULL;
	if( mSelectedStatic )
		mSelectedStatic->setSelectedOnFrame( frame );
	mPickStatic.reset();

	// pick things
	mSelectedThing = (CJammerThingEntity*)mPickThing.getMinColData();
	if( mPickStatic.getMinDistance() < mPickThing.getMinDistance() )
		mSelectedThing = NULL;
	if( mSelectedThing )
		mSelectedThing->setSelectedOnFrame( frame );
	mPickThing.reset();

	// pick tags
	mSelectedTag = (CJammerTagInstance*)mPickTag.getMinColData();
	if( mSelectedTag ) {
		mSelectedTag->setSelectedOnFrame( frame );
		((CJammerStaticEntity*)mSelectedTag->getOwner())->setIndicatedOnFrame( frame );
	}
	mPickTag.reset();

	if( mSelectedTagForLink && mEditMode == MODE_LINK ) {
		mSelectedTagForLink->setSelectedOnFrame( frame );
		((CJammerStaticEntity*)mSelectedTagForLink->getOwner())->setSelectedOnFrame( frame );
	}

	CGameInfo::getInstance().getLevel().update();
	mJamLevel->update();
	mCamera->activate();
}


void CJammer::linkStaticTags( CJammerTagInstance& tag1, CJammerTagInstance& tag2 ) const
{
	CJammerStaticEntity* e1 = (CJammerStaticEntity*)tag1.getOwner();
	CJammerStaticEntity* e2 = (CJammerStaticEntity*)tag2.getOwner();
	SMatrix4x4 tagInverse = tag2.getTag().getMatrix();
	tagInverse.invert();

	SMatrix4x4 tagDest = tag1.getTag().getMatrix() * e1->getWorldMatrix();
	float flipX = tag2.getFlips().x*tag1.getFlips().x;
	float flipY = tag2.getFlips().y*tag1.getFlips().y;
	float flipZ = tag2.getFlips().z*tag1.getFlips().z;
	tagDest.getAxisX() *= flipX;
	tagDest.getAxisY() *= flipY;
	tagDest.getAxisZ() *= flipZ;
	bool flip = e1->getCullMode() == D3DCULL_CW;
	if( flipX < 0 ) flip = !flip;
	if( flipY < 0 ) flip = !flip;
	if( flipZ < 0 ) flip = !flip;
	e2->setCullMode( flip ? D3DCULL_CW : D3DCULL_CCW );

	e2->getMatrix() = tagInverse * tagDest;
	e2->getWorldMatrix() = e2->getMatrix();
	tag2.getWorldMatrix() = tag2.getTag().getMatrix() * e2->getWorldMatrix();

	//
	// mark tags as linked and visited
	tag1.setLink( &tag2 );
	tag2.setLink( &tag1 );
	tag1.setVisitedOnFrame( mLinkFrameCounter );
	tag2.setVisitedOnFrame( mLinkFrameCounter );

	//
	// recurse on entity2 tags
	moveStaticEntity( *e2 );
}


// --------------------------------------------------------------------------


void CPickColListener::onCollide( dingus::CCollidable& me, dingus::CCollidable& him )
{
	if( !mActive )
		return;
	
	// category check
	if( !(him.getCategoryBits() & mPickCategoryBits) )
		return;

	// final collide check, get possible contact
	dContactGeom contacts[1];
	int ncontacts = dCollide( me, him, 1, contacts, sizeof(dContactGeom) );
	if( ncontacts < 1 )
		return;

	// get distance, see if nearer
	if( contacts[0].depth < mMinDistance ) {
		mMinPosition.set( contacts[0].pos[0], contacts[0].pos[1], contacts[0].pos[2] );
		mMinNormal.set( contacts[0].normal[0], contacts[0].normal[1], contacts[0].normal[2] );
		mMinDistance = contacts[0].depth;
		mMinColData = him.getUserData();
	}
}

void CPickColListener::reset()
{
	mMinDistance = 1.0e6f;
	mMinColData = NULL;
}
