// --------------------------------------------------------------------------
// Unco project - an entity/component framework on top of Dingus project
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __PARTICLES_COMP_H
#define __PARTICLES_COMP_H

#include "AbstractGfxComponent.h"
#include "../../kernel/Contexts.h"

#include <dingus/kernel/Proxies.h>

#include <dingus/gfx/particles/ParticleSystem.h>
#include <dingus/gfx/particles/VBParticleRenderer.h>
#include <dingus/gfx/geometry/VBManagerSource.h>
#include <dingus/renderer/RenderableBuffer.h>


namespace unco {



template< typename PARTICLE >
class CParticlesComponent : public CAbstractGfxComponent, public dingus::CRenderListenerAdapter {
public:
	typedef dingus::CParticleSystem<PARTICLE>	TParticleSystem;
	typedef dingus::IParticlePhysics<PARTICLE>	TParticlePhysics;
	typedef dingus::CAbstractParticleRenderer<PARTICLE>	TParticleRenderer;

public:
	CParticlesComponent( TParticlePhysics& physics, TParticleRenderer& renderer,
		dingus::CD3DIndexBuffer& ib,
		dingus::CResourceId const& fxID, int renderLayer, const dingus::SMatrix4x4& worldMatrix );
	CParticlesComponent( TParticlePhysics& physics, TParticlePhysics& postUpdatePhysics, TParticleRenderer& renderer,
		dingus::CD3DIndexBuffer& ib,
		dingus::CResourceId const& fxID, int renderLayer, const dingus::SMatrix4x4& worldMatrix );
	virtual ~CParticlesComponent();

	// CRenderListenerAdapter
	virtual void beforeRender( dingus::CRenderable& r, dingus::CEffectSlot const& fx );

	TParticleSystem const& getParticleSystem() const { return mParticleSystem; };
	TParticleSystem& getParticleSystem() { return mParticleSystem; };

	// CAbstractComponent
	virtual void update();

private:
	TParticleSystem		mParticleSystem;
	TParticleRenderer*	mParticleRenderer;
	dingus::CD3DIndexBuffer*	mIB;

	dingus::CRenderableIndexedBuffer*		mRenderableIBuffer;
	//CVBChunkProxy*							mVBChunk;
};


// --------------------------------------------------------------------------
//  Implementation


template< typename PARTICLE >
CParticlesComponent<PARTICLE>::CParticlesComponent( TParticlePhysics& physics, TParticleRenderer& renderer, dingus::CD3DIndexBuffer& ib, dingus::CResourceId const& fxID, int renderLayer, const dingus::SMatrix4x4& worldMatrix )
:	CAbstractGfxComponent( fxID, renderLayer, worldMatrix ),
	mParticleSystem( physics ),
	mParticleRenderer( &renderer ),
	mIB( &ib ),
	mRenderableIBuffer( NULL )
{
	mRenderableIBuffer = new dingus::CRenderableIndexedBuffer();
	setRenderable( *mRenderableIBuffer );
	getRenderable().addListener( *this );
}

template< typename PARTICLE >
CParticlesComponent<PARTICLE>::CParticlesComponent( TParticlePhysics& physics, TParticlePhysics& postUpdatePhysics, TParticleRenderer& renderer, dingus::CD3DIndexBuffer& ib, dingus::CResourceId const& fxID, int renderLayer, const dingus::SMatrix4x4& worldMatrix )
:	CAbstractGfxComponent( fxID, renderLayer, worldMatrix ),
	mParticleSystem( physics, postUpdatePhysics ),
	mParticleRenderer( &renderer ),
	mIB( &ib ),
	mRenderableIBuffer( NULL )
{
	mRenderableIBuffer = new dingus::CRenderableIndexedBuffer();
	setRenderable( *mRenderableIBuffer );
	getRenderable().addListener( *this );
}

template< typename PARTICLE >
CParticlesComponent<PARTICLE>::~CParticlesComponent()
{
	// delete mVBChunk
	delete mRenderableIBuffer;
}

template< typename PARTICLE >
void CParticlesComponent<PARTICLE>::update()
{
	mParticleSystem.update();
}

template< typename PARTICLE >
void CParticlesComponent<PARTICLE>::beforeRender( dingus::CRenderable& r, dingus::CEffectSlot const& fx )
{
	assert( mRenderableIBuffer );
	assert( mIB );

	mRenderableIBuffer->resetVBs();

	if( mParticleSystem.getParticleCount() < 1 )
		return;


	// render

	dingus::CRenderCamera& camera = CContexts::getRenderCtx().getCamera();
	dingus::CVBManagerSource vbSource( CContexts::getDynamicVBManager(), mParticleRenderer->getStride() );

	mParticleRenderer->begin(
		camera.getEye3(), camera.getCameraRotMatrix(),
		vbSource, mParticleSystem.getParticleCount()
	);
	mParticleSystem.render( *mParticleRenderer );
	dingus::CVBChunk::TSharedPtr chunk = mParticleRenderer->end();

	// to slot

	mRenderableIBuffer->setIB( *mIB );
	mRenderableIBuffer->setVB( chunk->getVB(), 0 );
	mRenderableIBuffer->setStride( chunk->getStride(), 0 );

	mRenderableIBuffer->setBaseVertex( chunk->getOffset() );
	mRenderableIBuffer->setMinVertex( 0 );
	mRenderableIBuffer->setNumVertices( chunk->getSize() );
	mRenderableIBuffer->setStartIndex( 0 );
	mRenderableIBuffer->setPrimCount( chunk->getSize()/2 );
	mRenderableIBuffer->setPrimType( mParticleRenderer->getPrimitiveType() );
}



}; // namespace


#endif
