// --------------------------------------------------------------------------
// Unco project - an entity/component framework on top of Dingus project
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#ifndef __TYPED_NAMED_CONTAINER_H
#define __TYPED_NAMED_CONTAINER_H

#include "NamedEntity.h"


namespace unco {


template<typename T>
class CTypedNamedContainer : public CNamedEntity {
public:
	typedef boost::intrusive_ptr<T>				TEntityPtr;
	typedef std::map<std::string,TEntityPtr>	TEntityMap;
	
public:
	CTypedNamedContainer( const std::string& name ) : CNamedEntity(name) { };

	virtual void update();
	virtual void attach();
	virtual void detach();
	
	void	addEntity( const TEntityPtr& entity );
	void	removeEntity( const TEntityPtr& entity );
	TEntityPtr findEntity( const std::string& name ) const;
	void	renameEntity( const TEntityPtr& entity, const std::string& newName );

	TEntityMap const& getEntities() const { return mEntities; }
	TEntityMap& getEntities() { return mEntities; }

protected:
	virtual void internalPostAttach() { }
	virtual void internalPreDetach() { }

private:
	TEntityMap	mEntities;
};


template<typename T>
void CTypedNamedContainer<T>::addEntity( const TEntityPtr& entity )
{
	assert( !entity->hasContainer() || !"already has container" );
	mEntities.insert( std::make_pair( entity->getName(), entity ) );
	entity->setContainer( this );
}

template<typename T>
void CTypedNamedContainer<T>::removeEntity( const TEntityPtr& entity )
{
	assert( entity->hasContainer() || !"no container" );
	assert( &entity->getContainer() == this || !"not in this container" );
	mEntities.erase( entity->getName() );
	entity->setContainer( NULL );
}

template<typename T>
void CTypedNamedContainer<T>::renameEntity( const TEntityPtr& entity, const std::string& newName )
{
	removeEntity( entity );
	entity->setName( newName );
	addEntity( entity );
}

template<typename T>
CTypedNamedContainer<T>::TEntityPtr CTypedNamedContainer<T>::findEntity( const std::string& name ) const
{
	TEntityMap::const_iterator it = mEntities.find( name );
	return (it == mEntities.end()) ? TEntityPtr(0) : it->second;
}


template<typename T>
void CTypedNamedContainer<T>::update()
{
	// update
	CEntity::update();

	// update children
	TEntityMap::iterator it, itEnd = mEntities.end();
	for( it = mEntities.begin(); it != itEnd; /* */ ) {
		TEntityPtr& e = it->second;
		assert( e );
		if( e->isDiscarded() ) {
			it = mEntities.erase( it );
			itEnd = mEntities.end();
		} else {
			e->update();
			++it;
		}
	}
}

template<typename T>
void CTypedNamedContainer<T>::attach()
{
	// attach self
	CEntity::attach();

	// attach children
	TEntityMap::iterator it, itEnd = mEntities.end();
	for( it = mEntities.begin(); it != mEntities.end(); ++it ) {
		assert( it->second );
		it->second->attach();
	}

	// final post attach
	internalPostAttach();
}

template<typename T>
void CTypedNamedContainer<T>::detach()
{
	// pre detach
	internalPreDetach();

	// detach children
	TEntityMap::iterator it, itEnd = mEntities.end();
	for( it = mEntities.begin(); it != mEntities.end(); ++it ) {
		assert( it->second );
		it->second->detach();
	}

	// detach self
	CEntity::detach();
}


};

#endif
