// --------------------------------------------------------------------------
// Unco project - an entity/component framework on top of Dingus project
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "CameraController.h"

using namespace unco;

// --------------------------------------------------------------------------
//  utils
// --------------------------------------------------------------------------


/**
 *  Axis to axis quaternion double angle (no normalization).
 *  Takes two points on unit sphere at angle THETA apart, returns
 *  quaternion that represents a rotation around cross product by 2*THETA.
 */
inline void gQuatUnitAxisToUnitAxis2(
									 dingus::SQuaternion& out,
									 const dingus::SVector3& from, const dingus::SVector3& to )
{
	dingus::SVector3 axis = from.cross( to ); // proportional to sin(theta)
	out.x = axis.x;
	out.y = axis.y;
	out.z = axis.z;
	out.w = from.dot(to);
}

/**
 *  Axis to axis quaternion.
 *  Takes two points on unit sphere at angle THETA apart, returns
 *  quaternion that represents a rotation around cross product by theta.
 */
inline void gQuatAxisToAxis( dingus::SQuaternion& out,
							const dingus::SVector3& from, const dingus::SVector3& to )
{
	dingus::SVector3 a, b;
	a = from.getNormalized();
	b = to.getNormalized();
	dingus::SVector3 half(a + b);
	half.normalize();
	gQuatUnitAxisToUnitAxis2( out, a, half );
}



// --------------------------------------------------------------------------
//  camera controller
// --------------------------------------------------------------------------

CCameraController::CCameraController()
:	mAffectedEntity(NULL),
	mArcBallRadius( 2.0f ),
	mArcBallDistance( 10.0f )
{
	mArcBallQuatBefore.identify();
	mArcBallQuatNow.identify();
	mArcBallVecBegin.set(0,0,0);
}


dingus::SVector3 CCameraController::arcBallScreenToVector( float x, float y ) const
{
	x *=  mArcBallRadius;
	y *= -mArcBallRadius;
	
	float z   = 0.0f;
	float mag = x*x + y*y;
	
	if( mag > 1.0f ) {
		float scale = 1.0f/sqrtf(mag);
		x *= scale;
		y *= scale;
	} else
		z = sqrtf( 1.0f - mag );
	
	return dingus::SVector3( x, y, z );
}

void CCameraController::arcBallBeginRotate( float x, float y )
{
	if( !mAffectedEntity )
		return;

	mArcBallVecBegin = arcBallScreenToVector( x, y );
	mArcBallQuatBefore = mArcBallQuatNow;
	mArcBallMatBefore = mAffectedEntity->getMatrix();
}

void CCameraController::arcBallRotate( float x, float y )
{
	if( !mAffectedEntity )
		return;

	// recompute mArcBallQuatNow
	dingus::SVector3 vecNow = arcBallScreenToVector( x, y );
	dingus::SQuaternion q;
	gQuatAxisToAxis( q, mArcBallVecBegin, vecNow );
	mArcBallQuatNow = mArcBallQuatBefore;
	mArcBallQuatNow *= q;
	mArcBallVecBegin = vecNow;
	dingus::SMatrix4x4& m = mAffectedEntity->getMatrix();
	//m = mArcBallMatBefore;
	// move forward
	m.getOrigin() += m.getAxisZ() * mArcBallDistance;
	// rotate
	dingus::SMatrix4x4 mrot;
	D3DXMatrixRotationQuaternion( &mrot, &q );
	m = mrot * m;
	// move backward
	m.getOrigin() -= m.getAxisZ() * mArcBallDistance;
}

void CCameraController::arcBallTranslate( float x, float y, float z )
{
	if( !mAffectedEntity )
		return;

	dingus::SMatrix4x4& m = mAffectedEntity->getMatrix();
	m.getOrigin() += m.getAxisX() * x;
	m.getOrigin() += m.getAxisY() * y;
	m.getOrigin() += m.getAxisZ() * z;
}
