// --------------------------------------------------------------------------
// Unco project - an entity/component framework on top of Dingus project
// Developed by nesnausk! team: www.nesnausk.org
// --------------------------------------------------------------------------

#include "../stdafx.h"
#pragma hdrstop

#include "System.h"
#include "EngineApplication.h"
#include "Contexts.h"
#include "AppContext.h"

#include <dingus/dxutils/D3DFont.h>

#include <dingus/kernel/SystemClock.h>
#include <dingus/kernel/D3DDevice.h>

#include <dingus/resource/SharedTextureBundle.h>
#include <dingus/resource/MeshBundle.h>
#include <dingus/resource/TextureBundle.h>
#include <dingus/resource/EffectBundle.h>
#include <dingus/resource/VertexDeclBundle.h>
#include <dingus/resource/IndexBufferBundle.h>
#include <dingus/resource/DeviceResource.h>
#include <dingus/resource/ReloadableBundle.h>

#include <dingus/renderer/RenderContext.h>
#include <dingus/renderer/RenderStage.h>
#include <dingus/visibility/VisibilityContext.h>
#include <dingus/visibility/VisibilityStage.h>
#include <dingus/input/InputContext.h>
#include <dingus/input/InputStage.h>
#include <dingus/input/DIKeyboard.h>
#include <dingus/input/DIMouse.h>

#include <dingus/pipeline/Pipeline.h>

#include <dingus/gfx/geometry/DynamicVBManager.h>

#include <dingus/console/W32StdConsoleRenderingContext.h>
#include <dingus/console/WDebugConsoleRenderingContext.h>
#include <dingus/console/D3DConsoleRenderingContext.h>


using namespace unco;


/**
 *  Constructor. Paired with ~CSystem().
 *  Member variables should be initialized to a known state here. The
 *  application window has not yet been created and no Direct3D device has been
 *  created, so any initialization that depends on a window or Direct3D should
 *  be deferred to a later stage. 
 */
CSystem::CSystem( IEngineApplication& application )
:	mApplication( &application ),
	mAppInited( false )
{
	SStartupParams params = mApplication->getStartupParams();
	mCreationWidth		= params.windowWidth;
	mCreationHeight 	= params.windowHeight;
	mWindowTitle		= params.windowTitle;
	mStartFullscreen	= params.startFullscreen;
	mShowCursorWhenFullscreen			= params.showCursorFullscreen;
	mEnumeration.mUsesDepthBuffer		= params.usesZBuffer;
	mEnumeration.mMinColorChannelBits	= params.minColorBits;
	mEnumeration.mMinAlphaChannelBits	= params.minAlphaBits;
	mEnumeration.mMinDepthBits			= params.minZBits;
	mEnumeration.mMinStencilBits		= params.minStencilBits;
	mDataPath = params.dataPath;

	mFont = new dingus::CD3DFont( "Arial", 10, dingus::CD3DFont::BOLD );
};


CSystem::~CSystem()
{
	delete mFont;
}


/**
 *  Initialization. Paired with shutdown().
 *  The window has been created and the IDirect3D9 interface has been
 *  created, but the device has not been created yet. Here you can
 *  perform application-related initialization and cleanup that does
 *  not depend on a device.
 */
HRESULT CSystem::initialize()
{
	//
	// init console

	mStdConsoleCtx = new dingus::CWDebugConsoleRenderingContext();
	dingus::CConsole::getInstance().setDefaultRenderingContext( *mStdConsoleCtx );

	/**/
	dingus::CD3DTextBoxConsoleRenderingContext* ctxSystem = 
		new dingus::CD3DTextBoxConsoleRenderingContext( *mFont, 2, 2, 0x60FFFF80 );
	
	mD3DConsoleCtxs.push_back( ctxSystem );

	dingus::CConsole::getChannel( "system" ).setRenderingContext( *ctxSystem );
	/**/
	
	//
	// resources

	const std::string basePath = mDataPath;
	dingus::CTextureBundle::init( basePath + "tex/" );
	dingus::CMeshBundle::init( basePath + "mesh/" );
	dingus::CEffectBundle::init( basePath + "fx/" );
	dingus::CSharedTextureBundle::init();
	dingus::CVertexDeclBundle::init();
	dingus::CIndexBufferBundle::init();

	dingus::CDynamicVBManager *vbManager = new dingus::CDynamicVBManager( 2 * 1024 * 1024 ); // 2 megabytes

	//
	// device dependant resources

	mDeviceManager = new dingus::CDeviceResourceManager();
	mDeviceManager->add( dingus::CSharedTextureBundle::getInstance() );
	mDeviceManager->add( *vbManager );
	mDeviceManager->add( dingus::CTextureBundle::getInstance() );
	mDeviceManager->add( dingus::CMeshBundle::getInstance() );
	mDeviceManager->add( dingus::CEffectBundle::getInstance() );
	mDeviceManager->add( dingus::CVertexDeclBundle::getInstance() );
	mDeviceManager->add( dingus::CIndexBufferBundle::getInstance() );

	//
	// reloadable resources

	mReloadableManager = new dingus::CReloadableBundleManager();
	mReloadableManager->add( dingus::CTextureBundle::getInstance() );
	mReloadableManager->add( dingus::CMeshBundle::getInstance() );
	mReloadableManager->add( dingus::CEffectBundle::getInstance() );

	//
	// contexts

	CContextRegistry& ctxs = CContextRegistry::getInstance();
	ctxs.setContext( CTX_RENDER, new dingus::CRenderContext() );
	ctxs.setContext( CTX_VIS, new dingus::CVisibilityContext() );
	ctxs.setContext( CTX_INPUT, new dingus::CInputContext() );
	ctxs.setContext( CTX_DYNAVB, vbManager );

	//
	// input devices

	HRESULT hr;

	IDirectInput8* directInput8 = NULL;
	hr = DirectInput8Create( GetModuleHandle(NULL), DIRECTINPUT_VERSION, IID_IDirectInput8, (void**)&directInput8, NULL );
	assert( SUCCEEDED( hr ) );
	assert( directInput8 );
	CContexts::getInputCtx().addDevice( *(new dingus::CDIKeyboard(mHWnd,*directInput8)) );
	CContexts::getInputCtx().addDevice( *(new dingus::CDIMouse(mHWnd,*directInput8)) );

	//
	// pipeline

	mPipeline = new dingus::CPipeline();
	dingus::IPipelineStage* stage;
	// application
	mPipeline->addStage( *mApplication );
	// input stage
	stage = new dingus::CInputStage( CContexts::getInputCtx() );
	mCreatedPipelineStages.push_back( stage );
	mPipeline->addStage( *stage );
	// visibility stage
	stage = new dingus::CVisibilityStage( CContexts::getVisCtx() );
	mCreatedPipelineStages.push_back( stage );
	mPipeline->addStage( *stage );
	// render stage
	stage = new dingus::CRenderStage( CContexts::getRenderCtx() );
	mCreatedPipelineStages.push_back( stage );
	mPipeline->addStage( *stage );

	return S_OK;
}


/**
 *  Called during device initialization, this code checks the display device
 *  for some minimum set of capabilities.
 */
HRESULT CSystem::checkDevice( const D3DCAPS9& caps, DWORD behavior, D3DFORMAT format )
{
	return mApplication->checkDevice( caps, behavior, format ) ? S_OK : E_FAIL;
}



/**
 *  createDeviceObjects(). Paired with deleteDeviceObjects().
 *  The device has been created.  Resources that are not lost on
 *  Reset() can be created here -- resources in D3DPOOL_MANAGED, D3DPOOL_SCRATCH,
 *  or D3DPOOL_SYSTEMMEM. Image surfaces created via CreateImageSurface are never
 *  lost and can be created here. Vertex shaders and pixel shaders can also be
 *  created here as they are not lost on Reset().
 */
HRESULT CSystem::createDeviceObjects()
{
	dingus::gD3DDevice = mD3DDevice;

	mFont->createDeviceObjects( *mD3DDevice );
	
	CAppContext::getInstance().setInfo( mBackBuffer.Width, mBackBuffer.Height, mFrameStats, mDeviceStats );

	mDeviceManager->createResource();

	if( !mAppInited ) {
		mApplication->initialize();
		mAppInited = true;
	}

	CContexts::getRenderCtx().notifyChanged();

	return S_OK;
}


/**
 *  activateDeviceObjects(). Paired with passivateDeviceObjects().
 *  The device exists, but may have just been Reset(). Resources in
 *  D3DPOOL_DEFAULT and any other device state that persists during rendering
 *  should be set here.
 */
HRESULT CSystem::activateDeviceObjects()
{
	mFont->activateDeviceObjects();
	CAppContext::getInstance().setInfo( mBackBuffer.Width, mBackBuffer.Height, mFrameStats, mDeviceStats );
	mDeviceManager->activateResource();
	return S_OK;
}


/**
 *  Called once per frame, the call is the entry point for 3d rendering. This
 *  function sets up render states, clears the viewport, and renders the scene.
 */
HRESULT CSystem::performOneTime()
{
	dingus::CSystemClock& c = dingus::CSystemClock::getInstance();
	c.setTimes( mTime, mElapsedTime, c.getPerformCount()+1 );
	CAppContext::getInstance().setInfo( mBackBuffer.Width, mBackBuffer.Height, mFrameStats, mDeviceStats );

	//
	// pipeline

	CContexts::getRenderCtx().getStats().reset();

	mPipeline->perform();

	//
	// stats

	dingus::CConsoleChannel& cc = dingus::CConsole::getChannel( "system" );
	cc.write( mFrameStats );
	cc.write( mDeviceStats );
	dingus::CRenderStats const& stats = CContexts::getRenderCtx().getStats();
	char buf[200];
	sprintf( buf, "Draw calls: %i  Vertices: %i  Prims: %i", stats.getDrawCalls(), stats.getVerticesRendered(), stats.getPrimsRendered() );
	cc.write( buf );
	
	dingus::gD3DDevice->BeginScene();
	TD3DConsoleCtxVector::iterator it, itEnd = mD3DConsoleCtxs.end();
	for( it = mD3DConsoleCtxs.begin(); it != itEnd; ++it ) {
		(*it)->flush();
	}
	dingus::gD3DDevice->EndScene();
	
	return S_OK;
}


/**
 *  Overrrides the main WndProc, so the sample can do custom message handling
 *  (e.g. processing mouse, keyboard, or menu commands).
 */
LRESULT CSystem::msgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
	static bool reloadKeyPressed = false;
	switch( msg ) {
	case WM_KEYDOWN:
		if( wParam == VK_F5 )
			reloadKeyPressed = true;
		break;
	case WM_KEYUP:
		if( wParam == VK_F5 ) {
			if( reloadKeyPressed ) {
				mReloadableManager->reload();
				CContexts::getRenderCtx().notifyChanged();
			}
			reloadKeyPressed = false;
		}
	}
	return CD3DApplication::msgProc( hWnd, msg, wParam, lParam );
}


/**
 *  Invalidates device objects.  Paired with activateDeviceObjects().
 */
HRESULT CSystem::passivateDeviceObjects()
{
	mFont->passivateDeviceObjects();
	mDeviceManager->passivateResource();
	return S_OK;
}

/**
 *  Paired with createDeviceObjects().
 *  Called when the app is exiting, or the device is being changed,
 *  this function deletes any device dependent objects.
 */
HRESULT CSystem::deleteDeviceObjects()
{
	mFont->deleteDeviceObjects();
	mDeviceManager->deleteResource();
	dingus::gD3DDevice = NULL;
	return S_OK;
}


/**
 *  Paired with initialize().
 *  Called before the app exits, this function gives the app the chance
 *  to cleanup after itself.
 */
HRESULT CSystem::shutdown()
{
	mApplication->shutdown();

	mDeviceManager->clear();
	delete mDeviceManager;

	mReloadableManager->clear();
	delete mReloadableManager;

	delete &CContexts::getRenderCtx();
	delete &CContexts::getVisCtx();
	delete &CContexts::getDynamicVBManager();
	dingus::stl_utils::wipe( CContexts::getInputCtx().getDevices() );
	delete &CContexts::getInputCtx();
	CContextRegistry::finalize();
	CAppContext::finalize();
	

	dingus::CTextureBundle::finalize();
	dingus::CMeshBundle::finalize();
	dingus::CEffectBundle::finalize();
	dingus::CVertexDeclBundle::finalize();
	dingus::CSharedTextureBundle::finalize();
	dingus::CIndexBufferBundle::finalize();

	dingus::stl_utils::wipe( mCreatedPipelineStages );
	delete mPipeline;

	dingus::CSystemClock::finalize();
	dingus::stl_utils::wipe( mD3DConsoleCtxs );
	dingus::CConsole::finalize();

	return S_OK;
}
