#include "../stdafx.h"
#pragma hdrstop

#include "Vehicle.h"
#include "CollideClasses.h"
#include <dingus/physics-ode/PhysicsContext.h>
#include <unco/kernel/Contexts.h>
#include <dingus/math/Constants.h>
#include <dingus/resource/TextureBundle.h>


CVehicle::CVehicle( const std::string& name, const CVehicleClass& clazz, unco::CVisibilityComponent& visComp )
:	unco::CTypedNamedContainer<CBasePhysicsEntity>(name), mClass(&clazz),
	mOutOfBounds(false),
	mAmmo(15)
{
	mColContainer = new dingus::CSimpleCollidableContainer( NULL );

	// create sub entities
	const float BODY_MASS = 500.0f;
	const float WHEEL_MASS = 50.0f;
	unsigned int catBits = 1<<COL_ENTITY_CAR; // CAR
	unsigned int colBits = (1<<COL_ENTITY_CAR) | (1<<COL_ENTITY_STATIC) | (1<<COL_ENTITY_THING) | (1<<COL_ENTITY_PICKUP);
	addEntity( new CBasePhysicsEntity( CVehicleClass::BODY_NAME, mClass->getBodyClass(), catBits, colBits, false, visComp, BODY_MASS, *mColContainer ) );

	// HACK
	getBody().getMeshes().front()->getParams().addTexture(
		"tEnv",
		*dingus::CTextureBundle::getInstance().getResourceById("envmap.png")
	);

	for( int i = 0; i < CVehicleClass::MAX_WHEELS; ++i ) {
		addEntity( new CBasePhysicsEntity( CVehicleClass::WHEEL_NAMES[i], mClass->getWheelClass((CVehicleClass::eWheel)i), catBits, colBits, false, visComp, WHEEL_MASS, *mColContainer ) );
	}

	// joint group
	mJoints = new dingus::CPhysicsJointGroup();
}

CVehicle::~CVehicle()
{
	delete mColContainer;
	delete mJoints;
}

void CVehicle::internalUpdate()
{
	getMatrix() = getBody().getWorldMatrix();
}

void CVehicle::internalAttach()
{
	// Position children into corrent positions; they will be attached after this
	// and physics stuff get initialized correctly. Will create joints later in
	// internalPostAttach()
	SMatrix4x4 mainM = getMatrix();
	getBody().getPhysComp().setMatrixPORed( mainM );
	for( int i = 0; i < CVehicleClass::MAX_WHEELS; ++i ) {
		CVehicleClass::eWheel w = (CVehicleClass::eWheel)i;
		unco::CPhysicsComponent& wheelP = getWheel(i).getPhysComp();
		const CTag& bodyWheelTag = mClass->getBodyWheelTag(w);
		const CTag& wheelTag = mClass->getWheelTag(w);
		SMatrix4x4 tagWMat = bodyWheelTag.getMatrix() * mainM;
		// TBD

		wheelP.setMatrixPORed( tagWMat );
	}
}

void CVehicle::internalDetach()
{
	mJoints->clear();
	for( int i = 0; i < CVehicleClass::MAX_WHEELS; ++i ) {
		delete mHinges[i];
	}
}

void CVehicle::internalPostAttach()
{
	dingus::CPhysicsContext& ctx = unco::CContexts::getPhysicsCtx();

	// create joints

	CBasePhysicsEntity& body = getBody();
	unco::CPhysicsComponent& bodyP = body.getPhysComp();
	for( int i = 0; i < CVehicleClass::MAX_WHEELS; ++i ) {
		CVehicleClass::eWheel w = (CVehicleClass::eWheel)i;
		CBasePhysicsEntity& wheel = getWheel(i);
		unco::CPhysicsComponent& wheelP = wheel.getPhysComp();
		mHinges[i] = new dingus::CPhysicsJointHinge2( ctx, *mJoints );
		mHinges[i]->attach( bodyP.getPhysicsable(), wheelP.getPhysicsable() );
		mHinges[i]->setAnchor( wheelP.getMatrix().getOrigin() );
		mHinges[i]->setAxis1( wheelP.getMatrix().getAxisY() );
		mHinges[i]->setAxis2( bodyP.getMatrix().getAxisX() );
		mHinges[i]->setLoStopAngle( 0 );
		mHinges[i]->setHiStopAngle( 0 );
		mHinges[i]->setMotorMaxForce( 3000 ); // 200
		mHinges[i]->setMotor2Velocity( 0 );
		mHinges[i]->setMotor2MaxForce( /*i>=2?500:1000*/1000 );
		mHinges[i]->setSuspensionERP( 0.15f ); // none
		mHinges[i]->setSuspensionCFM( 0.0007f ); // 0.0007
		wheelP.getPhysicsable().setFiniteRotationMode( true );
	}
}

void CVehicle::internalPreDetach()
{
}


void CVehicle::control( float velocity, float steering )
{
	if( mOutOfBounds )
		return;

	static const float steeringRate = (dingus::PI) * 4.0f / 3.0f;
	static const float steeringLimit = (dingus::PI) / 9.0f;

	float wheelVelocity = 22.0f * (dingus::PI) * velocity;

	// a little bit of deadband seems to make it a bit easier to control
	if( fabsf(steering) < 0.1f )
		steering = 0;

	int i;

	// steer
	for( i = 0; i < CVehicleClass::WHEEL_BR; ++i ) {
		float desiredPosition = steering;
		float actualPosition = mHinges[i]->getAngle1();
		float steeringVelocity = (desiredPosition - actualPosition) * 1.5f;

		mHinges[i]->setHiStopAngle( steeringLimit );
		mHinges[i]->setLoStopAngle( -steeringLimit );
		mHinges[i]->setMotorVelocity( steeringVelocity );
	}

	// drive - 4WD :)
	for( i = 0; i < CVehicleClass::MAX_WHEELS; ++i ) {
		mHinges[i]->setMotor2Velocity( -wheelVelocity );
	}
}
