#include "../stdafx.h"
#pragma hdrstop

#include "LevelSerializer.h"
#include <dingus/lua/LuaWrapper.h>
#include <dingus/lua/LuaValue.h>
#include <dingus/lua/LuaHelper.h>
#include <dingus/lua/LuaIterator.h>


// --------------------------------------------------------------------------
//  CLevelFileWriter
// --------------------------------------------------------------------------

CLevelFileWriter::CLevelFileWriter( const char* fileName )
{
	mFile = fopen( fileName, "wt" );
	assert( mFile );
}

CLevelFileWriter::~CLevelFileWriter()
{
	assert( mFile );
	fclose( mFile );
}

// --------------------------------------------------------------------------

void CLevelFileWriter::beginStaticEntities()
{
	assert( mFile );
	fprintf( mFile, "levelStatics = {\n" );
}

void CLevelFileWriter::writeStaticEntity(
	const std::string& name, const std::string& clazzName,
	const SMatrix4x4& matrix, bool flipCull )
{
	assert( mFile );
	fprintf( mFile, "{ name='%s', clazz='%s', pos={%g,%g,%g}, flipCull=%i,\n",
		name.c_str(), clazzName.c_str(), matrix._41, matrix._42, matrix._43, flipCull?1:0 );
	fprintf( mFile, "\trot={%g,%g,%g,%g,%g,%g,%g,%g,%g} },\n",
		matrix._11, matrix._12, matrix._13,
		matrix._21, matrix._22, matrix._23,
		matrix._31, matrix._32, matrix._33 );
}

void CLevelFileWriter::endStaticEntities()
{
	assert( mFile );
	fprintf( mFile, "}\n\n" );
}

// --------------------------------------------------------------------------

void CLevelFileWriter::beginThingEntities()
{
	assert( mFile );
	fprintf( mFile, "levelThings = {\n" );
}

void CLevelFileWriter::writeThingEntity(
	const std::string& name, const std::string& clazzName,
	const std::string& staticName, const SMatrix4x4& matrix )
{
	assert( mFile );
	fprintf( mFile, "{ name='%s', clazz='%s', static='%s', pos={%g,%g,%g},\n",
		name.c_str(), clazzName.c_str(), staticName.c_str(), matrix._41, matrix._42, matrix._43 );
	fprintf( mFile, "\trot={%g,%g,%g,%g,%g,%g,%g,%g,%g} },\n",
		matrix._11, matrix._12, matrix._13,
		matrix._21, matrix._22, matrix._23,
		matrix._31, matrix._32, matrix._33 );
}

void CLevelFileWriter::endThingEntities()
{
	assert( mFile );
	fprintf( mFile, "}\n\n" );
}

// --------------------------------------------------------------------------

void CLevelFileWriter::beginStaticLinks()
{
	assert( mFile );
	fprintf( mFile, "levelStaticLinks = {\n" );
}

void CLevelFileWriter::writeStaticLink(
	const std::string& entity1Name, const std::string& entity2Name,
	const std::string& tag1Name, const std::string& tag2Name,
	const SVector3& tag1flips, const SVector3& tag2flips )
{
	assert( mFile );
	fprintf( mFile, "{ e1name='%s', e2name='%s', tag1name='%s', tag2name='%s', flips1={%i,%i,%i}, flips2={%i,%i,%i} },\n",
		entity1Name.c_str(), entity2Name.c_str(), tag1Name.c_str(), tag2Name.c_str(),
		tag1flips.x>0 ? 1:-1, tag1flips.y>0 ? 1:-1, tag1flips.z>0 ? 1:-1,
		tag2flips.x>0 ? 1:-1, tag2flips.y>0 ? 1:-1, tag2flips.z>0 ? 1:-1 );
}

void CLevelFileWriter::endStaticLinks()
{
	assert( mFile );
	fprintf( mFile, "}\n\n" );
}



// --------------------------------------------------------------------------
//  CLevelFileReader
// --------------------------------------------------------------------------

void CLevelFileReader::read( const char* fileName, ILevelCreator& creator )
{
	using dingus::CLuaValue;
	using dingus::CLuaWrapper;
	using dingus::CLuaHelper;
	using dingus::CLuaArrayIterator;

	CLuaWrapper lua( "" );
	lua.doFile( fileName );

	//
	// read static objects

	CLuaValue luaStats = lua.getGlobal( "levelStatics" );
	CLuaArrayIterator itStats(luaStats);
	while( itStats.hasNext() ) {
		CLuaValue& luaStat = itStats.next();
		assert( luaStat.isTable() );

		// name, class, flipCull
		std::string statName = CLuaHelper::getString(luaStat,"name");
		std::string statClazz = CLuaHelper::getString(luaStat,"clazz");
		bool flipCull = ((int)CLuaHelper::getNumber(luaStat,"flipCull"))?true:false;

		// pos, rot
		SMatrix4x4 m;
		m.identify();
		m.getOrigin() = CLuaHelper::getVector3(luaStat,"pos");
		CLuaHelper::getMatrix3x3(luaStat,"rot",m);

		// notify creator
		creator.createStaticEntity( statName, statClazz, m, flipCull );
	}
	lua.discard(); // luaStats

	//
	// read static tags

	CLuaValue luaTags = lua.getGlobal( "levelStaticLinks" );
	CLuaArrayIterator itTags(luaTags);
	while( itTags.hasNext() ) {
		CLuaValue& luaTag = itTags.next();
		assert( luaTag.isTable() );

		// e1name, e2name, tag1name, tag2name
		std::string tagE1Name = CLuaHelper::getString(luaTag,"e1name");
		std::string tagE2Name = CLuaHelper::getString(luaTag,"e2name");
		std::string tagT1Name = CLuaHelper::getString(luaTag,"tag1name");
		std::string tagT2Name = CLuaHelper::getString(luaTag,"tag2name");

		// flips1, flips2
		SVector3 flips1 = CLuaHelper::getVector3(luaTag,"flips1");
		SVector3 flips2 = CLuaHelper::getVector3(luaTag,"flips2");

		// notify creator
		creator.createStaticLink( tagE1Name, tagE2Name, tagT1Name, tagT2Name, flips1, flips2 );
	}
	lua.discard(); // luaTags

	//
	// read thing objects

	CLuaValue luaThings = lua.getGlobal( "levelThings" );
	CLuaArrayIterator itThings(luaThings);
	while( itThings.hasNext() ) {
		CLuaValue& luaThing = itThings.next();
		assert( luaThing.isTable() );

		// name, class, static
		std::string thingName = CLuaHelper::getString(luaThing,"name");
		std::string thingClazz = CLuaHelper::getString(luaThing,"clazz");
		std::string thingStatic = CLuaHelper::getString(luaThing,"static");

		// pos, rot
		SMatrix4x4 m;
		m.identify();
		m.getOrigin() = CLuaHelper::getVector3(luaThing,"pos");
		CLuaHelper::getMatrix3x3(luaThing,"rot",m);

		// notify creator
		creator.createThingEntity( thingName, thingClazz, thingStatic, m );
	}
	lua.discard(); // luaThings
}

