#!/usr/bin/python
import pygame
import random
from pygame.locals import FULLSCREEN, RESIZABLE

INITIAL_SCREEN_WIDTH = 1024
INITIAL_SCREEN_HEIGHT = 768

FIGURE_SIZE = 40
LEVEL_OFFSET = (0, 300)
ENEMY_OFFSET = (800, 60)
ENEMY_TARGET = (200, 60)
ENEMY_TIME = 525

LAUNCH_BUTTON_OFFSET = (560, 640)
LAUNCH_BUTTON_SIZE = (200, 100)

BACKGROUND_COLOUR = (0, 80, 120)
DOODAD_COLOUR = (196, 196, 196)
FONT_COLOUR = (196, 196, 196)

FIGURE_BORDER = 5
FIGURE_OFFSET = (902, 325)

NAME_OFFSETS = ((160, 40),
                (400, 40))

FF_OFFSET = (1, 660)
FF_SIZE = (30, 100)

GAME_START = object()
GAME_RUNNING = object()
GAME_KILL = object()
GAME_HELP = object()
GAME_FAIL = object()
GAME_WIN = object()

figures = {}

# j - ###
#       #
figures['j'] = [(0, 0), (1, 0), (2, 0), (2, 1)]

# l - ###
#     #
figures['l'] = [(0, 0), (1, 0), (2, 0), (0, 1)]

# t - ###
#      #
figures['t'] = [(0, 0), (1, 0), (2, 0), (1, 1)]

# o - ##
#     ##
figures['o'] = [(0, 0), (1, 0), (1, 0), (1, 1)]

# i - ####
figures['i'] = [(0, 0), (1, 0), (2, 0), (3, 0)]

# x -  #
#     ###
#      #
figures['x'] = [(1, 0), (0, 1), (1, 1), (2, 1), (1, 2)]

# . - #
figures['dot'] = [(0, 0)]

# : - ##
figures['dotdot'] = [(0, 0), (1, 0)]

# s -  ##
#     ##
figures['s'] = [(1, 0), (2, 0), (0, 1), (1, 1)]

# z - ##
#      ##
figures['z'] = [(0, 0), (1, 0), (1, 1), (2, 1)]


def normalized(squares):
    mx = min(x for x, y in squares)
    my = min(y for x, y in squares)
    return [(x - mx, y - my) for x, y in squares]

def rotated(squares, n):
    if n == 0:
        return normalized(squares)
    else:
        squares = [(-y, x) for x, y in squares]
        return rotated(squares, n - 1)


class Figure(object):

    def __init__(self, template):
        self.image = pygame.image.load("images/figures/sprogmuo.png")
        self.image.set_colorkey((255, 255, 255))
        self._squares = template[1]
        self.rotation = 0

    @property
    def squares(self):
        return rotated(self._squares, self.rotation)

    def rotate(self):
        self.image = pygame.transform.rotate(self.image, -90)
        self.rotation += 1
        self.rotation = self.rotation % 4

    @property
    def width(self):
        return max([x for x, y in self.squares]) + 1

    @property
    def height(self):
        return max([y for x, y in self.squares]) + 1

    @property
    def pxWidth(self):
        return self.width * FIGURE_SIZE

    @property
    def pxHeight(self):
        return self.height * FIGURE_SIZE

    def render(self, surface, mx, my):
        ox, oy = self.pxWidth / 2, self.pxHeight / 2

        for x, y in self.squares:
            points = [(x, y),
                      (x + 1, y),
                      (x + 1, y + 1),
                      (x, y + 1)]
            points = [((x * FIGURE_SIZE) + mx - ox,
                       (y * FIGURE_SIZE) + my - oy) for x, y in points]
            color = DOODAD_COLOUR
            surface.blit(self.image, (x * FIGURE_SIZE + mx - ox,
                                      y * FIGURE_SIZE + my - oy - 40,))
            #pygame.draw.polygon(surface, color, points)
        #surface.blit(self.image, (mx - ox, my - oy))

    def drop(self, area, mx, my):
        area_ox = area.x * FIGURE_SIZE + LEVEL_OFFSET[0]
        area_oy = area.y * FIGURE_SIZE + LEVEL_OFFSET[1]
        offset_x, offset_y = self.pxWidth / 2, self.pxHeight / 2
        figure_ox = mx - offset_x
        figure_oy = my - offset_y

        # snap
        in_area_x = ((figure_ox - area_ox + (FIGURE_SIZE / 2)) / FIGURE_SIZE)
        in_area_y = ((figure_oy - area_oy + (FIGURE_SIZE / 2) ) / FIGURE_SIZE)

        if (in_area_x < 0 or in_area_y < 0):
            return False

        for x, y in self.squares:
            if (x + in_area_x >= area.w or
                y + in_area_y >= area.h):
                return False

        for ox, oy, figure in area.figures:
            for other in figure.squares:
                for square in self.squares:
                    if (other[0] + ox == square[0] + in_area_x and
                        other[1] + oy == square[1] + in_area_y):
                        return False
        area.figures.append((in_area_x, in_area_y, self))
        return True

    def outside(self, area, mx, my):
        area_ox = area.x * FIGURE_SIZE + LEVEL_OFFSET[0]
        area_oy = area.y * FIGURE_SIZE + LEVEL_OFFSET[1]
        offset_x, offset_y = self.pxWidth / 2, self.pxHeight / 2
        figure_ox = mx - offset_x
        figure_oy = my - offset_y

        # snap
        in_area_x = ((figure_ox - area_ox + (FIGURE_SIZE / 2)) / FIGURE_SIZE)
        in_area_y = ((figure_oy - area_oy + (FIGURE_SIZE / 2) ) / FIGURE_SIZE)

        for x, y in self.squares:
            if (0 < x + in_area_x < area.w and
                0 < y + in_area_y < area.h):
                return False
        return True

    def hit(self, px, py, mx, my):
        ox, oy = self.pxWidth / 2, self.pxHeight / 2
        for x, y in self.squares:
            points = [(x, y), (x + 1, y + 1)]
            points = [((x * FIGURE_SIZE) + px - ox,
                       (y * FIGURE_SIZE) + py - oy) for x, y in points]
            if (points[0][0] <= mx <= points[1][0] and
                points[0][1] <= my <= points[1][1]):
                return True

areas = {}
areas[4, 4] = (-57, -68, "images/levels/liemene_maza.png")
areas[5, 4] = (-69, -110, "images/levels/liemene_didele.png")
areas[3, 2] = (-20, -62, "images/levels/cemodanas.png")

class Area(object):

    def __init__(self, x, y, w, h):
        self.x, self.y, self.w, self.h = x, y, w, h
        self.figures = []
        self.image = pygame.image.load(areas[w, h][2])

    def render(self, surface):
        dx, dy = LEVEL_OFFSET

        xx = dx + self.x * FIGURE_SIZE + areas[self.w, self.h][0]
        yy = dy + self.y * FIGURE_SIZE + areas[self.w, self.h][1]
        surface.blit(self.image, (xx, yy))

        points = [(self.x, self.y),
                  (self.x + self.w, self.y),
                  (self.x + self.w, self.y + self.h),
                  (self.x, self.y + self.h)]
        points = [((x  * FIGURE_SIZE) + dx, (y  * FIGURE_SIZE) + dy) for x, y in points]
        color = DOODAD_COLOUR
        #pygame.draw.lines(surface, color, True, points)
        for x, y, figure in self.figures:
            figure.render(surface,
                          ((self.x + x)  * FIGURE_SIZE) + dx + figure.pxWidth / 2,
                          ((self.y + y)  * FIGURE_SIZE) + dy + figure.pxHeight / 2)

    @property
    def weight(self):
        return sum(len(figure.squares) for x, y, figure in self.figures)

levels = {}

levels['01'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4}],
                'bad_guy': ('jeep', 4),
                'ffs': 1}

levels['02'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4}],
                'bad_guy': ('jeep_with_gun', 8),
                'ffs': 1}

levels['03'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4}],
                'bad_guy': ('jeep_with_gun', 10),
                'ffs': 1}

levels['04'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 8, 'y': 6, 'w': 3, 'h': 2}],
                'bad_guy': ('jeep_with_gun', 12),
                'ffs': 1} # Armoured personell carrier

levels['05'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 8, 'y': 4, 'w': 4, 'h': 4}],
                'bad_guy': ('tank', 20),
                'ffs': 2}

levels['06'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 8, 'y': 4, 'w': 4, 'h': 4}],
                'bad_guy': ('tank', 24),
                'ffs': 2}

levels['07'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 9, 'y': 4, 'w': 5, 'h': 4}],
                'bad_guy': ('tank', 28),
                'ffs': 2}

levels['08'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 8, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 13, 'y': 6, 'w': 3, 'h': 2}],
                'bad_guy': ('tank', 28),
                'ffs': 2}

levels['09'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 8, 'y': 4, 'w': 4, 'h': 4}],
                'bad_guy': ('boss', 28),
                'ffs': 2}

levels['10'] = {'areas': [{'x': 2, 'y': 4, 'w': 4, 'h': 4},
                          {'x': 9, 'y': 4, 'w': 5, 'h': 4}],
                'bad_guy': ('boss', 32),
                'ffs': 2}

from name import generate

def load_image(file_name, surface):
    image = pygame.image.load(file_name)
    return image.convert_alpha(surface)

class Level(object):

    def __init__(self, template, surface):
        name, definition = template
        bad_guy_image, bad_guy_hp = definition['bad_guy']
        self.name = name
        # self.image = load_image("images/levels/%s.png" % name, surface)
        self.enemy = pygame.image.load("images/enemies/%s.png" % bad_guy_image)
        self.enemy.set_colorkey((255, 255, 255))
        self.areas = [Area(**kwargs)
                      for kwargs in definition['areas']]
        self.ffs = definition['ffs']
        self.names = [generate() for i in range(self.ffs)]
        self.enemy_hp = bad_guy_hp
        self.time_left = ENEMY_TIME

    def tick(self):
        self.time_left -= 1

    @property
    def ticks(self):
        return ENEMY_TIME - self.time_left

    @property
    def enemy_position(self):
        sx, sy = ENEMY_OFFSET
        tx, ty = ENEMY_TARGET
        return tx + (sx - tx) * (self.time_left / float(ENEMY_TIME))

    def render(self, surface):
        # surface.blit(self.image, LEVEL_OFFSET)

        sx, sy = ENEMY_OFFSET
        tx, ty = ENEMY_TARGET

        y = sy
        x = self.enemy_position
        surface.blit(self.enemy, (x, y))

        for area in self.areas:
            area.render(surface)
        font = pygame.font.Font("freesansbold.ttf", 40)
        text = font.render(self.name, 1, (0, 0, 0))
        surface.blit(text, (600 - 10 - text.get_width(), 300))

        for (x, y), name in zip(NAME_OFFSETS, self.names):
            small_font = pygame.font.Font("freesansbold.ttf", 22)
            text = small_font.render(name, 1, (0, 0, 0))
            surface.blit(text, (x - (text.get_width() / 2),
                                y + LEVEL_OFFSET[1]))

        text = font.render(":%02d" % (self.time_left / 25), 1, (0, 0, 0))
        surface.blit(text, (surface.get_width() - 10 - text.get_width(), 10))

class App(object):

    quit = False
    state = GAME_START

    def __init__(self):
        self.fullscreen = True
        self.width, self.height = INITIAL_SCREEN_WIDTH, INITIAL_SCREEN_HEIGHT
        self.set_mode()
        self.start_screen = load_image("images/game_start.png", self.surface)
        self.help_screen = load_image("images/game_help.png", self.surface)
        self.win_screen = load_image("images/game_win.png", self.surface)
        self.fail_screen = load_image("images/game_fail.png", self.surface)
        self.launch_button = load_image("images/launch.png", self.surface)
        self.game = load_image("images/game.png", self.surface)
        self.hp = load_image("images/hp.png", self.surface)
        self.ff = load_image("images/ff.png", self.surface)
        self.update_state = self.start
        self.active_figure = None

    def set_mode(self):
        if self.fullscreen:
            w, h = max(pygame.display.list_modes(), key=lambda (w, h): w*h)
            flags = FULLSCREEN
        else:
            w, h = self.width, self.height
            flags = RESIZABLE
        self.surface = pygame.display.set_mode((w, h), flags)

    def toggle_fullscreen(self):
        self.fullscreen = not self.fullscreen
        self.set_mode()

    def update(self, events):
        for event in events:
            if event.type == pygame.QUIT:
                self.quit = True
            elif event.type == pygame.VIDEORESIZE:
                self.resize_window(event.size) # XXX  TODO
            elif event.type == pygame.KEYDOWN and event.key == pygame.K_q:
                self.quit = True
            elif event.type == pygame.KEYDOWN and event.key == pygame.K_ESCAPE:
                self.quit = True
            elif event.type == pygame.KEYDOWN and event.key == pygame.K_f:
                self.toggle_fullscreen()
        self.update_state(events)

    def show_help(self):
        self.old_state = self.state
        self.old_update = self.update_state
        self.state = GAME_HELP
        self.update_state = self.help

    def start_game(self):
        self.kill_ticks = -1
        self.active_level = Level(sorted(levels.items())[0], self.surface)
        self.level = 0
        self.fighters = 5
        self.figures = [self.getRandomFigure(),
                        self.getRandomFigure(),
                        self.getRandomFigure(),
                        self.getRandomFigure()]
        self.active_figure = None
        self.state = GAME_RUNNING
        self.update_state = self.run
        self.names = []
        self.time = 0

    def game_win(self):
        extra_names = 10
        extra_names += 5 - self.fighters
        self.names.extend([generate() for i in range(extra_names)])
        self.state = GAME_WIN
        self.update_state = self.win

    def game_fail(self):
        extra_names = self.level
        extra_names += 5
        self.names.extend([generate() for i in range(extra_names)])
        self.state = GAME_FAIL
        self.update_state = self.fail

    def start(self, events):
        for event in events:
            if event.type == pygame.KEYDOWN and event.key == pygame.K_F1:
                self.show_help()
            if event.type == pygame.MOUSEBUTTONUP:
                self.start_game()

    def getRandomFigure(self):
        return Figure(random.choice(list(figures.items())))

    def kaboom(self):
        self.update_state = self.kill
        self.state = GAME_KILL

    kill_ticks = -1

    def kill(self, events):
        if self.kill_ticks == -1:
            self.kill_ticks = 0
        else:
            self.kill_ticks += 1

        position = self.kill_ticks * 20
        self.draw_level()
        self.surface.blit(self.ff, (position, 180))

        if position < self.active_level.enemy_position + 50:
            return

        total = sum([area.weight
                     for area in self.active_level.areas])
        self.active_level.enemy_hp -= total
        if self.active_level.enemy_hp > 0:
            self.fighters -= self.active_level.enemy_hp

        if self.fighters < 0:
            self.names.extend(self.active_level.names)
            self.time += self.active_level.ticks
            self.game_fail()
            return
        else:
            self.level += 1

        if self.level == 10:
            self.game_win()
        else:
            self.names.extend(self.active_level.names)
            self.time += self.active_level.ticks
            self.active_level = Level(sorted(levels.items())[self.level], self.surface)
            self.state = GAME_RUNNING
            self.update_state = self.run
        self.kill_ticks = -1

    def run(self, events):
        for event in events:
            if event.type == pygame.KEYDOWN and event.key == pygame.K_F1:
                self.show_help()
            if event.type == pygame.MOUSEBUTTONUP and event.button == 3:
                if self.active_figure:
                    self.active_figure.rotate()
            if event.type == pygame.MOUSEBUTTONUP and event.button == 2:
                self.active_level.enemy_hp = 0
            if (event.type == pygame.MOUSEBUTTONUP or
                event.type == pygame.MOUSEBUTTONDOWN) and event.button == 1:
                if self.active_figure is None:
                    # check if you have clicked on a figure
                    for x, y, figure in list(self.figure_positions()):
                        if figure.hit(x, y, *pygame.mouse.get_pos()):
                            self.active_figure = figure
                    if self.active_figure is None:
                        mx, my = pygame.mouse.get_pos()
                        lbx, lby = LAUNCH_BUTTON_OFFSET
                        lbw, lbh = LAUNCH_BUTTON_SIZE
                        if (lbx < mx < lbx + lbw and
                            lby < my < lby + lbh):
                            self.kaboom()
                else:
                    outside = True
                    for area in self.active_level.areas:
                        if self.active_figure.drop(area, *pygame.mouse.get_pos()):
                            self.figures.remove(self.active_figure)
                            self.figures.append(self.getRandomFigure())
                            self.active_figure = None
                            outside = False
                            break
                        elif not self.active_figure.outside(area, *pygame.mouse.get_pos()):
                            outside = False
                    if outside:
                        self.active_figure.rotation = 0
                        self.active_figure = None
        self.active_level.tick()
        if self.active_level.time_left == 0:
            self.kaboom()

    def help(self, events):
        for event in events:
            if ((event.type == pygame.KEYDOWN and event.key == pygame.K_F1) or
                (event.type == pygame.MOUSEBUTTONUP)):
                self.state = self.old_state
                self.update_state = self.old_update

    def fail(self, events):
        for event in events:
            if event.type == pygame.KEYDOWN and event.key == pygame.K_F1:
                self.show_help()
            if event.type == pygame.MOUSEBUTTONUP:
                self.start_game()

    def win(self, events):
        for event in events:
            if event.type == pygame.KEYDOWN and event.key == pygame.K_F1:
                self.show_help()
            if event.type == pygame.MOUSEBUTTONUP:
                self.start_game()

    def figure_positions(self):
        sx, sy = FIGURE_OFFSET
        for n, figure in enumerate(self.figures):
            yield sx, sy + n * (FIGURE_SIZE * 3 + FIGURE_BORDER), figure

    def draw_level(self):
        self.surface.blit(self.game, (0, 0))
        self.active_level.render(self.surface)
        for x, y, figure in self.figure_positions():
            if figure is not self.active_figure:
                figure.render(self.surface, x, y)
        ox, oy = FF_OFFSET
        w, h = FF_SIZE
        for x in range(self.fighters):
            self.surface.blit(self.ff, ((ox + x) * w, oy))
        offset = (200 - self.active_level.enemy_hp * 5) / 2
        for x in range(self.active_level.enemy_hp):
            self.surface.blit(self.hp, (self.active_level.enemy_position + x * 5 + offset, 70))
        if self.active_figure:
            self.active_figure.render(self.surface, *pygame.mouse.get_pos())

    def renderText(self, text, size, x, y):
        font = pygame.font.Font("freesansbold.ttf", size)
        gap = 0
        for line in text.splitlines():
            line_text = font.render(line, 1, (200, 200, 200))
            self.surface.blit(line_text,
                              (x - (line_text.get_width() / 2),
                               y + gap))
            gap += line_text.get_height() + 3

    def render(self):
        if self.state == GAME_START:
            self.surface.fill((0, 0, 0))
            text = """
Our Country was invaded by the enemy forces. Our army has lost the
war, and now the forces of the resistance must hide in caves, and use
guerilla tactics to combat the vastly superior enemy army. But if we
can win this war of attrition - peace and prosperity will come back to
our country. All we have to do - is sacrifice everything we have to
achieve victory!
"""
            info = """
Click   - Start
F1       - Help 
ESC     - Quit 
"""
            self.renderText(text, 24, 1024 / 2, 100)
            self.renderText(info, 40, 1024 / 2, 500)
            #self.surface.blit(self.start_screen, (0, 0))
        elif self.state == GAME_RUNNING:
            self.draw_level()
        elif self.state == GAME_HELP:
            self.surface.fill((0, 0, 0))
            info = """
Click - Pick up
Release - Drop
ESC   - Quit
Right click - Rotate
"""
            self.renderText(info, 40, 1024 / 2, 300)
            #self.surface.blit(self.help_screen, (0, 0))
        elif self.state == GAME_FAIL:
            self.surface.fill((0, 0, 0))
            self.renderText("K.I.A.", 40, 1024 / 2, 40)
            names = self.names
            soldiers1 = "\n".join(names[:len(names) / 2])
            soldiers2 = "\n".join(names[len(names) / 2:])

            self.renderText(soldiers1, 20, 1024 / 3, 90)
            self.renderText(soldiers2, 20, 1024 - 1024 / 3, 90)

            text = """
You have failed on your mission, all you did was take away sons and
daughters from the citizens of your country. And for what? To make
a point?! Well - I sure hope you are happy now.
"""
            info = """
Click - We will never surrender!
ESC   - Happy
"""
            self.renderText(text, 24, 1024 / 2, 400)
            self.renderText(info, 40, 1024 / 2, 500)
            #self.surface.blit(self.fail_screen, (0, 0))
        elif self.state == GAME_WIN:
            self.surface.fill((0, 0, 0))
            self.renderText("K.I.A.", 40, 1024 / 2, 40)
            names = self.names
            soldiers1 = "\n".join(names[:len(names) / 2])
            soldiers2 = "\n".join(names[len(names) / 2:])

            self.renderText(soldiers1, 20, 1024 / 3, 90)
            self.renderText(soldiers2, 20, 1024 - 1024 / 3, 90)

            text = """
Congratulations it took you only %s seconds to get all these people
killed. I sure hope it was worth it.
""" % self.time
            info = """
Click - Let's do it again
ESC   - Quit
"""
            self.renderText(text, 24, 1024 / 2, 400)
            self.renderText(info, 40, 1024 / 2, 500)
            #self.surface.blit(self.win_screen, (0, 0))
        font = pygame.font.Font("freesansbold.ttf", 15)
        text = font.render("FPS: %.1f" % self.clock.get_fps(), 1, FONT_COLOUR)
        scr_w = self.surface.get_width()
        scr_h = self.surface.get_height()
        self.surface.blit(text, (scr_w - 10 - text.get_width(),
                                 scr_h - 10 - text.get_height()))

def main():
    pygame.init()
    pygame.display.set_caption('Alfa Freedom Fighter')
    clock = pygame.time.Clock()
    app = App()
    app.clock = clock
    while not app.quit:
        app.update(pygame.event.get())
        app.render()
        pygame.display.flip()
        # wait for the next frame
        clock.tick(25)


if __name__ == '__main__':
    try:
        main()
    finally:
        pygame.quit()

