#!/usr/bin/env python
# -*- encoding: utf-8 -*-

# By Vytautas Šaltenis (Vytautas.Shaltenis@gmail.com)
# for Global Game Jam 2009, jan30-feb1

# based on the following:

#  -------------------------------
# pyODE example 3: Collision detection

# Originally by Matthias Baas.
# Updated by Pierre Gay to work without pygame or cgkit.

import sys, os, random, time
from math import *
from OpenGL.GL import *
from OpenGL.GLU import *
from OpenGL.GLUT import *

import pygame
import pygame.mixer
from pygame.locals import *

if not pygame.font: print 'Warning, fonts disabled'
if not pygame.mixer: print 'Warning, sound disabled'

import ode
import pdb

# A list with ODE bodies
bodies = []
contactgroup = None
world = None
space = None
lasttime = None

# Some variables used inside the simulation loop
fps = 50
dt = 1.0 / fps
counter = 0
particleRadius = 0.1
playerRadius = 0.3
player = None
girl = None

whistleSound = None
femaleSounds = []
maleSounds = []

position = [0, 15, 0]   # eye position
lightPos = [5, 15, 0, 0]

def initOpenGL ():
    """Prepare drawing.
    """

    # Viewport
    glViewport (0, 0, 1024, 768)

    # Initialize
    glClearColor (0.8, 0.8, 0.9, 0)
    glClearDepth (1.0)
    glDepthMask (GL_TRUE)
    glDepthFunc (GL_LESS)
    glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
    glEnable (GL_DEPTH_TEST)
    glDisable (GL_LIGHTING)
    glEnable (GL_LIGHTING)
    glEnable (GL_NORMALIZE)
    glShadeModel (GL_SMOOTH)

    glEnable (GL_CULL_FACE)

    # Projection
    glMatrixMode (GL_PROJECTION)
    glLoadIdentity ()
    gluPerspective (45, 1.3333, 0.2, 20)

    # Initialize ModelView matrix
    glMatrixMode (GL_MODELVIEW)
    glLoadIdentity ()

    #global_ambient = [0.5, 0.5, 0.5, 1.0]
    #glLightModelfv (GL_LIGHT_MODEL_AMBIENT, global_ambient)
    glLightModeli (GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE)
    #glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE)

    # Light source
    glLightfv (GL_LIGHT0, GL_POSITION, lightPos)
    glLightfv (GL_LIGHT0, GL_AMBIENT, [0.5, 0.5, 0.5, 1])
    glLightfv (GL_LIGHT0, GL_DIFFUSE, [1, 1, 1, 1])
    glLightfv (GL_LIGHT0, GL_SPECULAR, [1, 1, 1, 1])
    glEnable (GL_LIGHT0)

    # View transformation
    """
    gluLookAt (2.4, 3.6, 4.8,   # eye
               0.5, 0.5, 0,     # reference point
               0, 1, 0)         # up
    """
    gluLookAt (position[0], position[1], position[2],
               0, 0, 0,     # reference point
               0, 0, -1)    # up vector

def drawPlayer (body):
    glutSolidSphere (body.r, 10, 10)

    glPushMatrix ()
    glRotatef (135, 0, 1, 0)      # vector to rotate around
    glScalef (body.r / 2, body.r / 2, body.r * 2)
    glTranslatef (0, 0.5, body.r * 2)
    glutSolidCube (1)
    glPopMatrix ()

    glPushMatrix ()
    glRotatef (135, 0, 1, 0)      # vector to rotate around
    glScalef (body.r / 2, body.r / 2, body.r * 2)
    glTranslatef (0, 0.5, body.r * 3)
    glutSolidCone (1, 0.75, 10, 1)
    glPopMatrix ()

def drawGirl (body):
    glutSolidSphere (body.r, 10, 10)

    glPushMatrix ()
    glScalef (body.r / 2, body.r / 2, body.r * 2)
    glTranslatef (0, 0.5, body.r * 2)
    glutSolidCube (1)
    glPopMatrix ()

    glPushMatrix ()
    glRotatef (90, 0, 1, 0)      # vector to rotate around
    glScalef (body.r / 2, body.r / 2, body.r * 2)
    glTranslatef (-body.r * 9, 0.5, 0)
    glutSolidCube (1)
    glPopMatrix ()

def loadSound (name):
    class NoneSound:
        def play (self): pass

    if not pygame.mixer:
        return NoneSound ()

    try:
        sound = pygame.mixer.Sound (name)
    except pygame.error, message:
        print 'Cannot load sound:', name
        raise SystemExit, message

    return sound

def drawBody (body):
    """Draw an ODE body.
    """

    x, y, z = body.getPosition ()
    rot = [1, 0, 0, 0,
           0, 1, 0, 0,
           0, 0, 1, 0,
           x, y, z, 1]
    glPushMatrix ()
    glMultMatrixd (rot)

    if body.shape == 'sphere':
        glutSolidSphere (body.r, 10, 10)

    if body.shape == 'player':
        drawPlayer (body)

    if body.shape == 'girl':
        drawGirl (body)

    glPopMatrix ()

def createSphere (world, space, density, radius):
    """Create a sphere body and its corresponding geom."""

    # Create body
    body = ode.Body (world)
    M = ode.Mass ()
    M.setSphere (density, radius)
    body.setMass (M)

    # Set parameters for drawing the body
    body.shape = 'sphere'

    # Create a sphere geom for collision detection
    geom = ode.GeomSphere (space, radius)
    geom.setBody (body)

    return body

def dropObject ():
    """Drop an object into the scene."""
    global bodies, counter, particleRadius, world, space

    body = createSphere (world, space, 1000, particleRadius)
    body.setPosition ((random.gauss (-4, 4), 1.0, random.gauss (-4, 4)))
    body.r = particleRadius
    theta = random.uniform (0,2 * pi)
    ct = cos (theta)
    st = sin (theta)
    body.setRotation ([ct, 0., -st, 0., 1., 0., st, 0., ct])
    bodies.append (body)
    counter = 0

# Collision callback
def collisionCallback (args, geom1, geom2):
    """Callback function for the collide() method.

    This function checks if the given geoms do collide and
    creates contact joints if they do.
    """

    # Check if the objects do collide
    contacts = ode.collide (geom1, geom2)

    # Create contact joints
    world, contactgroup = args
    for c in contacts:
        c.setBounce (0.2)
        c.setMu (5000)
        j = ode.ContactJoint (world, contactgroup, c)
        j.attach (geom1.getBody (), geom2.getBody ())

        b1 = geom1.getBody ()
        b2 = geom2.getBody ()

        if b1 and b2:
            if b1.shape in ['player', 'girl'] and b2.shape in ['player', 'girl']:
                channel = pygame.mixer.Channel (0)
                channel.play (random.choice (maleSounds))
                channel.queue (random.choice (femaleSounds))

def addSpheres (numSpheres):
    for i in range (numSpheres):
        dropObject ()

    global bodies, player, girl, playerRadius, world, space

    player = createSphere (world, space, 1000, playerRadius)
    player.r = playerRadius
    player.shape = 'player'
    player.setPosition ((0, 0, 0))
    bodies.append (player)

    girl = createSphere (world, space, 1000, playerRadius)
    girl.r = playerRadius
    girl.shape = 'girl'
    girl.setPosition ((-3, -3, 0))
    bodies.append (girl)

def moveThingsAround ():
    global bodies

    factor = 80
    for b in bodies:
        if b.shape == 'sphere':
            l = b.getPosition ()
            l = [10 * random.uniform (-factor, factor),
                 10 * random.uniform (-factor, factor),
                 10 * random.uniform (-factor, factor)]
            b.addForce (l)

######################################################################

# keyboard callback
def _keyfunc (c, x, y):
    global player

    playerSpeed = 0.2
    pos = list (player.getPosition ())

    if c == 'w':
        position[1] -= 1
    elif c == 's':
        position[1] += 1
    elif c == 'j':
        pos[2] += playerSpeed
    elif c == 'k':
        pos[2] -= playerSpeed
    elif c == 'h':
        pos[0] -= playerSpeed
    elif c == 'l':
        pos[0] += playerSpeed

    player.setPosition (pos)

    if c == 'g':
        channel = pygame.mixer.Channel (0)
        channel.play (random.choice (maleSounds))
        channel.queue (random.choice (femaleSounds))

    if c == 'q':
        sys.exit (0)

# draw callback
def _drawfunc ():
    # Draw the scene
    initOpenGL ()

    for b in bodies:
        drawBody (b)

    glutSwapBuffers ()

# idle callback
def _idlefunc ():
    global counter, lasttime, dt, space

    t = dt - (time.time () - lasttime)
    if (t > 0):
        time.sleep (t)

    counter += 1
    moveThingsAround ()
    glutPostRedisplay ()

    # Simulate
    n = 2

    for i in range (n):
        # Detect collisions and create contact joints
        space.collide ((world, contactgroup), collisionCallback)

        # Simulation step
        world.step (dt / n)

        # Remove all contact joints
        contactgroup.empty ()

    lasttime = time.time ()

def loadFilesByList (list):
    sounds = []

    for f in list:
        sounds.append (loadSound (f))

    return sounds

def loadFemaleSounds ():
    global femaleSounds

    files = ['giggle.wav',          # from here: http://www.freesound.org/samplesViewSingle.php?id=58650
             'giggle2.wav',         # http://www.freesound.org/samplesViewSingle.php?id=15288
             'giggle3.wav',         # http://www.freesound.org/samplesViewSingle.php?id=15288
             'tonight.wav',         # from here: http://www.freesound.org/samplesViewSingle.php?id=4057
             'hisigh.wav',          # high sigh: http://www.freesound.org/samplesViewSingle.php?id=19695
             'losigh.wav',          # low sigh: http://www.freesound.org/samplesViewSingle.php?id=19696
             'grrr.aiff',           # from here: http://www.freesound.org/samplesViewSingle.php?id=18022
             'denied.aif',          # http://www.freesound.org/samplesViewSingle.php?id=23997
             'omg.aif',             # http://www.freesound.org/samplesViewSingle.php?id=66252
             'sneeze.wav',          # http://www.freesound.org/samplesViewSingle.php?id=54505
             'meow2.wav']           # http://www.freesound.org/samplesViewSingle.php?id=16466

    femaleSounds = loadFilesByList (files)

def loadMaleSounds ():
    global maleSounds

    files = ['onemore.wav',         # from here: http://www.freesound.org/samplesViewSingle.php?id=20560
             'whooah.wav',          # from here: http://www.freesound.org/samplesViewSingle.php?id=20565
             'humm1.wav',           # http://www.freesound.org/samplesViewSingle.php?id=22090
             'humm2.wav',           # http://www.freesound.org/samplesViewSingle.php?id=22090
             'humm3.wav']           # http://www.freesound.org/samplesViewSingle.php?id=22090

    maleSounds = loadFilesByList (files)

def main ():
    global contactgroup, lasttime, world, space

    # Initialize Glut
    glutInit ([])

    pygame.mixer.pre_init (22050, -16, 2, 512)
    pygame.init()
    #pygame.mixer.init ()

    # Open a window
    glutInitDisplayMode (GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH)

    x = 0
    y = 0
    width = 1024
    height = 768
    glutInitWindowPosition (x, y);
    glutInitWindowSize (width, height);
    glutCreateWindow ('Brown\'s Motion')

    # Create a world object
    world = ode.World ()
    world.setGravity ((0, -9.81, 0))
    world.setERP (0.8)
    world.setCFM (1E-5)

    # Create a space object
    space = ode.Space ()

    # Create a plane geom which prevent the objects from falling forever
    floor = ode.GeomPlane (space, (0, 1, 0), 0)
    ceiling = ode.GeomPlane (space, (0, -1, 0), -1)
    left = ode.GeomPlane (space, (1, 0, 0), -5)
    right = ode.GeomPlane (space, (-1, 0, 0), -5)
    near = ode.GeomPlane (space, (0, 0, -1), -5)
    far = ode.GeomPlane (space, (0, 0, 1), -5)

    loadFemaleSounds ()
    loadMaleSounds ()

    whistleSound = loadSound ('whistle.wav')
    # whistle is from here: http://www.freesound.org/samplesViewSingle.php?id=39548

    # A joint group for the contact joints that are generated whenever
    # two bodies collide
    contactgroup = ode.JointGroup ()

    lasttime = time.time ()
    addSpheres (80)

    glutKeyboardFunc (_keyfunc)
    glutDisplayFunc (_drawfunc)
    glutIdleFunc (_idlefunc)
    whistleSound.play ()
    glutMainLoop ()

if __name__ == '__main__':
    main ()

