#include "Defcon3D.h"
#include "Utils/Font.h"
#include "Camera/Camera.h"
#include "Camera/CFrustum.h"
#include "Utils/CKeys.h"

#include "Managers/CAnimationManager.hpp"
#include "Managers/CShaderManager.hpp"
#include "Managers/CTextureManager.hpp"
#include "Managers/CMaterialManager.hpp"
#include "Managers/CModelManager.hpp"
#include "Managers/CFrameBufferManager.hpp"
#include "Managers/CFXManager.hpp"
#include "Sounds/Audio.h"

#include "Lights/DefaultLight.hpp"
#include "Globals/Globals.hpp"

#include "AppUtils/CLoadingWnd.hpp"
#include "AppUtils/AppSettings.hpp"



#include "Props/CProperties.hpp"
#include "Props/CAttributeGroup.hpp"
#include "Props/Cattribute.hpp"
#include "AppListener.h"

#include "Planet.h"
#include "Tank.h"
#include "Trail.h"
#include "Rocket.h"
#include "Utils/CMouseState.h"
#include "Smoke.h"
#include "Primitives.h"

#include "DefconSettings.h"

#include "ErrorCheck.h"


Defcon3D::Defcon3D():OpenGLWnd(new AppListener())
{
	g_sApp.eAppType=PT_NORMAL;



    g_nViewportWd=512;
	g_nViewportHt=512;

}
Defcon3D::~Defcon3D()
{
}
bool	Defcon3D::InitManagers()
{
	if(!OpenGLWnd::InitManagers())return false;
	
	
	g_cLoadingWnd.Printf("Initializing ShaderManager");
	g_pcShaderMan=new CShaderManager();

	g_cLoadingWnd.Printf("Initializing TextureManager");
	g_pcTexMan=new CTextureManager();

	g_cLoadingWnd.Printf("Initializing MaterialManager");
	g_pcMatMan=new CMaterialManager();

	g_cLoadingWnd.Printf("Initializing ModelManager");
	g_pcModelMan=new CModelManager();
	
	g_cLoadingWnd.Printf("Initializing FXManager");
	g_pcFXMan=new CFXManager();

	g_cLoadingWnd.Printf("Initializing Sounds");
	g_pcSoundMan=new Audio();
	
	g_cLoadingWnd.Printf("Initializing FrameBuferrManager");
	g_pcFboMan = new CFrameBufferManager();
	
	g_pcAnimMan = new CAnimationManager();
	
	
	
	g_cLoadingWnd.Printf("Managers Loaded");
	
	g_pcShaderMan->SetDir("Data\\Shaders\\");
	
	g_pcTexMan->Init();
	g_pcFXMan->Init();
	g_pcMatMan->InitMaterials();
	
	
	
	return true;
}
void	Defcon3D::DeInitManagers()
{
	delete g_pcAnimMan;
	delete g_pcSoundMan;
	delete g_pcFXMan;
	delete g_pcModelMan;
	delete g_pcMatMan;
	delete g_pcTexMan;
	delete g_pcShaderMan;
	delete g_pcFboMan;
	
	OpenGLWnd::DeInitManagers();
}
bool	Defcon3D::LoadResources()
{
	g_cLoadingWnd.StartLoading(g_sApp.hInstance,g_sApp.hWnd);
	g_cLoadingWnd.Printf("Loading Started");
	
	if(!OpenGLWnd::LoadResources())return false;
	

	
	g_pcTexMan->SetDir("Data\\Textures\\font\\");
	if(!LoadFontTexture("font.tga")){
		MessageBox(NULL,"Failed to load font.tga","Error!",MB_OK);
		return false;
	}
	
	BuildFont();
	
	g_cLoadingWnd.EndLoading();

	g_cMainCam.SetSpeed(10);
	
	
	g_pcModelMan->SetDir("Data\\Models\\");
	g_pcMatMan->SetDir("Data\\Materials\\");
	g_pcTexMan->SetDir("Data\\Textures\\");

	m_pPlanet = new Planet();

	
	g_pcModelMan->LoadModel("Cannon.tct");
	m_pTanks[0] = new Tank(g_pcModelMan->LoadModel("RedTank.tct"));
	m_pTanks[1] = new Tank(g_pcModelMan->LoadModel("GreenTank.tct"));

	g_pcModelMan->LoadModel("Rocket.tct");

	m_nRocketCnt = Defcon::nRocketCnt;
	m_pRocket   = new Rocket*[m_nRocketCnt];
	for(uint i=0;i<m_nRocketCnt;i++)
	{
		m_pRocket[i]   = new Rocket();
	}

	m_pTrail    = new Trail(g_pcTexMan->LoadTexture2D("smoke.png"));


	m_pPlayer1  = g_pcTexMan->LoadTexture2D("Player1.png");
	m_pPlayer2  = g_pcTexMan->LoadTexture2D("Player2.png");
	m_pPower    = g_pcTexMan->LoadTexture2D("Power.png");
	m_pInstructions = g_pcTexMan->LoadTexture2D("Instructions.png");

	m_pVic1= g_pcTexMan->LoadTexture2D("Victory1.png");
	m_pVic2= g_pcTexMan->LoadTexture2D("Victory2.png");

	m_pTanks[0]->SetPosition(Vector3D(1,0,0).normalize()*100.0f);
	m_pTanks[0]->CalcRotation();
	m_pTanks[1]->SetPosition(Vector3D(-1,0,0).normalize()*100.0f);
	m_pTanks[1]->CalcRotation();

	m_pTargetTank = m_pTanks[0];
	m_nMaxSmokeSpawns = 20;

	m_pSmokeEmitters = SmokeEmitterPool::Create(m_nMaxSmokeSpawns,g_pcTexMan->LoadTexture2D("Smoke.png"));

	m_nSndFire = g_pcSoundMan->addSound("Data\\Sounds\\Fire.wav");
	m_nSndExplosion[0] = g_pcSoundMan->addSound("Data\\Sounds\\explosion2.wav");
	m_nSndExplosion[1] = g_pcSoundMan->addSound("Data\\Sounds\\explosion1.wav");
	m_nSndRocket = g_pcSoundMan->addSound("Data\\Sounds\\rocket.wav");

	

	m_nSndFireSrc = g_pcSoundMan->addSoundSource(m_nSndFire);
	m_nSndExplosionSrc[0] = g_pcSoundMan->addSoundSource(m_nSndExplosion[0]);
	m_nSndExplosionSrc[1] = g_pcSoundMan->addSoundSource(m_nSndExplosion[1]);
	m_nSndRocketSrc = g_pcSoundMan->addSoundSource(m_nSndRocket,LOOPING);


	m_vRocketsMidPos  = Vector3D(0,0,0);
	m_bSwitchingState = false;
	m_eTargetState    = STATE_NONE;
	m_fFadeValue	  = 0.0f;
	m_fWaitForExplosion= 0.0f;
	m_fPower		  = 0.0f;
	m_fTime			  = 0.0f;
	SetState(STATE_TITLE,false);

	
	return true;
}
void	Defcon3D::UnLoadResources()
{
	delete m_pSmokeEmitters;
	delete m_pTanks[0];
	delete m_pTanks[1];
	for(uint i=0;i<m_nRocketCnt;i++)delete m_pRocket[i];
	delete[]m_pRocket;
	delete m_pPlanet;
	KillFont();
	OpenGLWnd::UnLoadResources();
}	

void	Defcon3D::LoadSettings()
{
	CAttributeGroup* pcAttrGroup = g_pcAppProps->FindAttrGroup("Camera");

	g_cMainCam.PositionCamera(
		pcAttrGroup->FindAttribute("v3Eye")->GetVector3D(),
		pcAttrGroup->FindAttribute("v3At")->GetVector3D(),
		Vector3D(0,0,1));

	CAttributeGroup* pcGame = g_pcAppProps->FindAttrGroup("Game");
	Defcon::nRocketCnt				= iClamp(pcGame->FindAttribute("iRocketCnt")->GetInt(),1,200);
	Defcon::fDamageRadius			= pcGame->FindAttribute("fDamageRadius")->GetFloat();
	Defcon::fDamageDepth			= pcGame->FindAttribute("fDamageDepth")->GetFloat();
	Defcon::fPlanetGravityForce		= pcGame->FindAttribute("fPlanetGravityForce")->GetFloat();
	Defcon::fRocketDamping			= pcGame->FindAttribute("fRocketDamping")->GetFloat();
}
void	Defcon3D::SaveSettings()
{
	CAttributeGroup* pcAttrGroup = g_pcAppProps->FindAttrGroup("Camera");
	pcAttrGroup->FindAttribute("v3Eye")->SetVector3D(g_cMainCam.GetEye());
	pcAttrGroup->FindAttribute("v3At")->SetVector3D(g_cMainCam.GetAt());

}

bool	Defcon3D::AreRocketsDead()
{
	bool bAllRocketsDead = true;
	for(uint i=0;i<m_nRocketCnt;i++)
	{
		if(m_pRocket[i]->IsAlive())
		{
			bAllRocketsDead = false;break;
		}
	}
	return bAllRocketsDead;
}
void	Defcon3D::CalculateScores()
{
	m_nScores[0] = m_pPlanet->GetPlayerOneLand();
	m_nScores[1] = m_pPlanet->GetPlayerTwoLand();
}
void	Defcon3D::CheckInput()
{
	OpenGLWnd::CheckInput();
}

void	Defcon3D::Update(float fDeltaTime)
{
	fDeltaTime*=2.0f;
	OpenGLWnd::Update(fDeltaTime);
	//g_cMainCam.CameraControls(fDeltaTime);
	//g_cMainCam.SetViewByMouse();
	m_fTime += fDeltaTime;
	
	if(m_bSwitchingState)
	{
		if(m_fFadeValue<0.5f)
		{
			m_fFadeValue = fMin( m_fFadeValue+fDeltaTime*1.4f,0.5f);
			if(m_fFadeValue==0.5f)
			{
				m_eState = m_eTargetState;
				if(m_eState==STATE_PLAYER1_TURN || m_eState==STATE_PLAYER2_TURN)
				{
					//Set camera
					Vector3D vTankView = m_pTargetTank->GetPosition();vTankView.normalize();
					vTankView*=150.0f;
					m_cPointCamera.SetTarget(m_pTargetTank->GetPosition(),m_pTargetTank->GetPosition()+vTankView);
					m_cPointCamera.Update(0.0f);

				}
			}
		}
		else
		{
			m_fFadeValue = fMin( m_fFadeValue+fDeltaTime*1.4f,1.0f);
			if(m_fFadeValue==1.0f)m_bSwitchingState = false;
		}
		return;
	}
	
	switch(m_eState)
	{
	case STATE_TITLE:
		if(g_cKeys.Pressed(VK_RETURN))
		{
			m_pPlanet->Reset();
			CalculateScores();
			SetState(STATE_PLAYER1_TURN);
		}
		break;	
	case STATE_PLAYER1_TURN:
		m_fPower = (sinf(m_fTime*6.0f)+1.0f)*0.5f;
		//Rotate around tank
		if(g_cMouse.Pressed(0))
		{
			Vector3D vTankView = g_cMainCam.GetEye()-m_pTargetTank->GetPosition();vTankView.normalize();
			vTankView*=150.0f;
			m_cPointCamera.SetTarget(m_pTargetTank->GetPosition(),m_pTargetTank->GetPosition()+vTankView);
		}
		if(g_cMouse.GetButton(0))
		{
			m_cPointCamera.Update(fDeltaTime);
		}


		//Rotate around world
		if(g_cMouse.Pressed(1))
		{
			Vector3D vWorldView = g_cMainCam.GetEye();vWorldView.normalize();
			vWorldView*=300.0f;
			m_cPointCamera.SetTarget(Vector3D(0,0,0),vWorldView);
		}
		if(g_cMouse.GetButton(1))
		{
			m_cPointCamera.Update(fDeltaTime);
		}
	
		//Fire
		if(g_cKeys.Pressed(' '))
		{
			for(uint i=0;i<m_nRocketCnt;i++)
			{
				float fLocPower = m_fPower*70.0f+40.0f;
				Vector3D vVel(0,fRandom(fLocPower-3.0f,fLocPower+3.0f),0);
				vVel=m_pTargetTank->GetCanonRot()*vVel+GetRandomVector()*5.0f;

				m_pRocket[i]->Fire(m_pTargetTank->GetCanonPos()+
					m_pTargetTank->GetCanonRot()*Vector3D(0,7,0)
					,vVel);
			}

			Vector3D vDir = g_cMainCam.GetAt()-g_cMainCam.GetEye();
			vDir.normalize();
			SetState(STATE_PLAYER1_FIRING,false);
			g_pcSoundMan->play(m_nSndFireSrc);
			g_pcSoundMan->play(m_nSndRocketSrc);

			m_fWaitForExplosion = 3.0f;

			m_cChaseCamera.SetTrackObject(&m_vRocketsMidPos);
		}

		m_pTargetTank->Update(fDeltaTime);

		break;
	case STATE_PLAYER2_TURN:
		m_pTargetTank->UpdateCPU(fDeltaTime);
		if(m_pTargetTank->CPUWantsToFire())
		{
			m_fPower = fRandom(0.9f,1.0f);
			for(uint i=0;i<m_nRocketCnt;i++)
			{
				float fLocPower = m_fPower*70.0f+40.0f;
				Vector3D vVel(0,fRandom(fLocPower-3.0f,fLocPower+3.0f),0);
				vVel=m_pTargetTank->GetCanonRot()*vVel+GetRandomVector()*5.0f;
				m_pRocket[i]->Fire(m_pTargetTank->GetCanonPos()+
					m_pTargetTank->GetCanonRot()*Vector3D(0,7,0)
					,vVel);
			}

			Vector3D vDir = g_cMainCam.GetAt()-g_cMainCam.GetEye();
			vDir.normalize();
			SetState(STATE_PLAYER2_FIRING,false);
			g_pcSoundMan->play(m_nSndFireSrc);
			g_pcSoundMan->play(m_nSndRocketSrc);

			m_fWaitForExplosion = 3.0f;

			m_cChaseCamera.SetTrackObject(&m_vRocketsMidPos);
		}

		break;
	case STATE_PLAYER1_FIRING:
	case STATE_PLAYER2_FIRING:
		{
			if(AreRocketsDead()==false)
			{

				m_vRocketsMidPos = Vector3D(0,0,0);
				for(uint i=0;i<m_nRocketCnt;i++)
				{
					m_vRocketsMidPos+=m_pRocket[i]->GetPosition();
				}
				m_vRocketsMidPos/=float(m_nRocketCnt);
			
				m_cChaseCamera.Update(fDeltaTime);
				for(uint i=0;i<m_nRocketCnt;i++)
				{
					if(m_pRocket[i]->IsAlive()==false)continue;
					m_pRocket[i]->Update(fDeltaTime);
					m_pTrail->UpdateSpawns(fDeltaTime,m_pRocket[i]->GetPosition());
					Vector3D vView = m_pRocket[i]->GetPosition();vView.normalize();
				
					if(m_pRocket[i]->IsAlive()==false)
					{
						Vector3D vDir = -m_pRocket[i]->GetColPoint();
						vDir.normalize();

						g_pcSoundMan->play(m_nSndExplosionSrc[rand()%2]);
						m_pPlanet->OnHit(m_pRocket[i]->GetColPoint(),vDir);

						SmokeEmitterPool::Iterator * pItem = m_pSmokeEmitters->Allocate();
						if(pItem)
						{
							pItem->pData->Reset(m_pRocket[i]->GetColPoint());
						}

						
						if(AreRocketsDead())
						{
							g_pcSoundMan->stop(m_nSndRocketSrc);
							CalculateScores();
						}
					}
				}
			}
			else
			{
				m_fWaitForExplosion = fMax(m_fWaitForExplosion-fDeltaTime,0.0f);
				if(m_fWaitForExplosion==0.0f)
				{
					if(m_nScores[0]==0)SetState(STATE_PLAYER2_VICTORY);
					else if(m_nScores[1]==0)SetState(STATE_PLAYER1_VICTORY);
					else if(m_eState==STATE_PLAYER1_FIRING)SetState(STATE_PLAYER2_TURN);
					else if(m_eState==STATE_PLAYER2_FIRING)SetState(STATE_PLAYER1_TURN);

					
				}
			}
		}
		break;
	case STATE_PLAYER1_VICTORY:
	case STATE_PLAYER2_VICTORY:
		if(g_cKeys.Pressed(VK_RETURN))SetState(STATE_TITLE);
		if(g_cKeys.Pressed(VK_ESCAPE))PostQuitMessage(0);
		break;
	case STATE_RESTART:
		break;
	}


	
	//Update smoke

	SmokeEmitterPool::Iterator	  * pPrevItem = NULL;
	for(SmokeEmitterPool::Iterator* pItem = m_pSmokeEmitters->Next(NULL);
		pItem!=NULL;
		pItem = m_pSmokeEmitters->Next(pItem))
	{
		pItem->pData->Update(fDeltaTime);
		if(pItem->pData->IsAlive()==false)
		{
			m_pSmokeEmitters->Delete(pItem);
			pItem = pPrevItem;
		}
		else
		{
			pPrevItem = pItem;
		}
	}

	m_pTrail->Update(fDeltaTime);
}



void	Defcon3D::PreRender()
{
	g_cMainCam.SetMatrix();
	
	RenderStaticScene();

	RenderDynamicScene();

}
void	Defcon3D::RenderStaticScene()
{
	CHECK_GL_ERROR();

	g_cFrustum.CalculateFrustum();


	m_pPlanet->Render();
	m_pTanks[0]->Render();
	m_pTanks[1]->Render();
	for(uint i=0;i<m_nRocketCnt;i++)
	{
		if(m_pRocket[i]->IsAlive())m_pRocket[i]->Render();
	}
	m_pTrail->Render();



	for(SmokeEmitterPool::Iterator* pItem = m_pSmokeEmitters->Next(NULL);
		pItem!=NULL;
		pItem = m_pSmokeEmitters->Next(pItem))
	{
		pItem->pData->Render();
	}


}
void	Defcon3D::RenderDynamicScene()
{
}



void	Defcon3D::PostRender()
{
	glViewport(0,0,g_nWd,g_nHt);
	glEnable(GL_TEXTURE_2D);
	glDisable(GL_LIGHTING);

	switch(m_eState)
	{
	case STATE_PLAYER1_TURN:
	case STATE_PLAYER2_TURN:
	case STATE_PLAYER1_FIRING:
	case STATE_PLAYER2_FIRING:
		//g_cConsole.At(130,20);
		//g_cConsole.Printf("Player1:%d",m_nScores[0]);
		//g_cConsole.At(g_nWd-200,20);
		//g_cConsole.Printf("Player2:%d",m_nScores[1]);
		break;
	case STATE_PLAYER1_VICTORY:
		break;
	case STATE_PLAYER2_VICTORY:
		break;
	case STATE_RESTART:
		break;

	}



	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	
	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glPushMatrix();										// Store The Projection Matrix
	glLoadIdentity();									// Reset The Projection Matrix
	glOrtho(0,g_nWd,0,g_nHt,-1,1);						// Set Up An Ortho Screen
	glMatrixMode(GL_MODELVIEW);							// Select The Modelview Matrix
	glPushMatrix();										// Store The Modelview Matrix
	glLoadIdentity();									// Reset The Modelview Matrix

	glColor4f(1,1,1,1);

	glDisable(GL_TEXTURE_2D);

	DrawLeftQuad(Vector2D(43,g_nHt-47),23,159,Vector3D(1,0,0),Vector3D(0,1,0),float(m_nScores[0])/100.0f);
	DrawLeftQuad(Vector2D(g_nWd-197,g_nHt-47),23,159,Vector3D(1,0,0),Vector3D(0,1,0),float(m_nScores[1])/100.0f);


	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

	glBindTexture(GL_TEXTURE_2D,m_pPlayer1->GetTextureId());
	DrawLeftQuad(Vector2D(20,g_nHt-40),80,200);

	glBindTexture(GL_TEXTURE_2D,m_pPlayer2->GetTextureId());
	DrawLeftQuad(Vector2D(g_nWd-220,g_nHt-40),80,200);

	if(m_eState==STATE_TITLE)
	{
		glBindTexture(GL_TEXTURE_2D,m_pInstructions->GetTextureId());
		DrawLeftQuad(Vector2D(20,g_nHt-450),800,900);
	}
	
	if(m_eState==STATE_PLAYER1_VICTORY)
	{
		glBindTexture(GL_TEXTURE_2D,m_pVic1->GetTextureId());
		DrawLeftQuad(Vector2D(20,g_nHt-450),800,900);
	}

	if(m_eState==STATE_PLAYER2_VICTORY)
	{
		glBindTexture(GL_TEXTURE_2D,m_pVic2->GetTextureId());
		DrawLeftQuad(Vector2D(20,g_nHt-450),800,900);
	}
	
	

	if(m_eState==STATE_PLAYER1_TURN || m_eState==STATE_PLAYER1_FIRING)
	{
		glDisable(GL_TEXTURE_2D);
		DrawLeftQuad(Vector2D(43,g_nHt-137),23,159,Vector3D(1,0.5,0),Vector3D(1,0.0f,0),m_fPower);

		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D,m_pPower->GetTextureId());
		DrawLeftQuad(Vector2D(20,g_nHt-130),80,200);
	}




	if(m_bSwitchingState)
	{

		glDisable(GL_TEXTURE_2D);

		glColor4f(1,1,1,sinf(m_fFadeValue*PI));
		glBegin(GL_QUADS);
			glVertex2i(0,0);
			glVertex2i(g_nWd,0);
			glVertex2i(g_nWd,g_nHt);
			glVertex2i(0,g_nHt);
		glEnd();
	


	}

	glEnable(GL_DEPTH_TEST);
	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glPopMatrix();										// Restore The Old Projection Matrix
	glMatrixMode(GL_MODELVIEW);							// Select The Modelview Matrix
	glPopMatrix();										// Restore The Old Projection Matrix
	glDisable(GL_BLEND);

	glEnable(GL_LIGHTING);

}

void	Defcon3D::SetState(EState eState,bool withFade)
{
	m_eTargetState = eState;

	switch(m_eTargetState)
	{
	case STATE_TITLE:
		m_pPlanet->Reset();
		CalculateScores();
		break;	
	case STATE_PLAYER1_TURN:
		m_pTargetTank = m_pTanks[0];
		break;
	case STATE_PLAYER2_TURN:
		m_pTargetTank = m_pTanks[1];
		m_pTargetTank->DecideMove();
		break;
	case STATE_PLAYER1_FIRING:
		break;
	case STATE_PLAYER2_FIRING:
		break;
	case STATE_PLAYER1_VICTORY:
		break;
	case STATE_PLAYER2_VICTORY:
		break;
	case STATE_RESTART:
		break;
	}

	if(withFade==false)m_eState=m_eTargetState;
	else
	{
		m_bSwitchingState = true;
		m_fFadeValue      = 0.0f;
	}
}
